/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package ai.preferred.regression.pe;

import ai.preferred.regression.io.CSVInputData;
import org.apache.commons.csv.CSVPrinter;
import org.kohsuke.args4j.Option;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.ArrayList;

public class AverageCell extends ProcessingElement {

  private static final Logger LOGGER = LoggerFactory.getLogger(AverageCell.class);

  @Option(name = "-c", aliases = {"--column"})
  private int column;

  @Option(name = "-d", aliases = {"--delimiter"})
  private String delimiter = ",";

  @Override
  protected void process(CSVInputData data, CSVPrinter printer) throws IOException {
    if (data.hasHeader()) {
      ArrayList<String> header = data.getHeader();
      header.add(header.remove(column));
      printer.printRecord(header);
    }

    for (ArrayList<String> record : data) {
      double sum = 0.0;
      final String[] values = record.get(column).split(delimiter);
      for (final String v : values) {
        try {
          sum += Double.parseDouble(v.trim());
        } catch (NumberFormatException e) {
          LOGGER.error("Unable to parse the number", e);
        }
      }
      record.remove(column);
      record.add(String.valueOf(sum / values.length));
      printer.printRecord(record);
    }
  }

  public static void main(String[] args) {
    parseArgsAndRun(AverageCell.class, args);
  }

}
