/*
 * Decompiled with CFR 0.152.
 */
package ai.promoted.delivery.client;

import ai.promoted.delivery.client.ApiFactory;
import ai.promoted.delivery.client.ApplyTreatmentChecker;
import ai.promoted.delivery.client.DefaultApiFactory;
import ai.promoted.delivery.client.Delivery;
import ai.promoted.delivery.client.DeliveryException;
import ai.promoted.delivery.client.DeliveryRequest;
import ai.promoted.delivery.client.DeliveryResponse;
import ai.promoted.delivery.client.Metrics;
import ai.promoted.delivery.client.Sampler;
import ai.promoted.delivery.client.SamplerImpl;
import ai.promoted.delivery.model.ClientInfo;
import ai.promoted.delivery.model.ClientType;
import ai.promoted.delivery.model.CohortArm;
import ai.promoted.delivery.model.CohortMembership;
import ai.promoted.delivery.model.DeliveryExecution;
import ai.promoted.delivery.model.DeliveryLog;
import ai.promoted.delivery.model.ExecutionServer;
import ai.promoted.delivery.model.LogRequest;
import ai.promoted.delivery.model.Request;
import ai.promoted.delivery.model.Response;
import ai.promoted.delivery.model.Timing;
import ai.promoted.delivery.model.TrafficType;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.logging.Level;
import java.util.logging.Logger;

public class PromotedDeliveryClient {
    private static final String SERVER_VERSION = "java.2.0.6";
    private static final Logger LOGGER = Logger.getLogger(PromotedDeliveryClient.class.getName());
    public static final long DEFAULT_DELIVERY_TIMEOUT_MILLIS = 250L;
    public static final long DEFAULT_METRICS_TIMEOUT_MILLIS = 3000L;
    public static final int DEFAULT_METRICS_THREAD_POOL_SIZE = 5;
    public static final int DEFAULT_MAX_REQUEST_INSERTIONS = 1000;
    private final Delivery sdkDelivery;
    private final Delivery apiDelivery;
    private final Metrics apiMetrics;
    private final Executor executor;
    private final ApplyTreatmentChecker applyTreatmentChecker;
    private final float shadowTrafficDeliveryRate;
    private final Sampler sampler;
    private boolean performChecks;
    private boolean blockingShadowTraffic;

    private PromotedDeliveryClient(String deliveryEndpoint, String deliveryApiKey, long deliveryTimeoutMillis, String metricsEndpoint, String metricsApiKey, long metricsTimeoutMillis, boolean warmup, Executor executor, int maxRequestInsertions, ApplyTreatmentChecker applyTreatmentChecker, ApiFactory apiFactory, float shadowTrafficDeliveryRate, Sampler sampler, boolean performChecks, boolean blockingShadowTraffic, boolean acceptGzip) {
        if (deliveryTimeoutMillis <= 0L) {
            deliveryTimeoutMillis = 250L;
        }
        if (metricsTimeoutMillis <= 0L) {
            metricsTimeoutMillis = 3000L;
        }
        if (executor == null) {
            executor = Executors.newSingleThreadExecutor();
        }
        if (apiFactory == null) {
            apiFactory = new DefaultApiFactory();
        }
        if (maxRequestInsertions <= 0) {
            maxRequestInsertions = 1000;
        }
        if (shadowTrafficDeliveryRate < 0.0f || shadowTrafficDeliveryRate > 1.0f) {
            throw new IllegalArgumentException("shadowTrafficDeliveryRate must be between 0 and 1");
        }
        if (sampler == null) {
            sampler = new SamplerImpl();
        }
        this.sampler = sampler;
        this.executor = executor;
        this.shadowTrafficDeliveryRate = shadowTrafficDeliveryRate;
        this.applyTreatmentChecker = applyTreatmentChecker;
        this.sdkDelivery = apiFactory.createSdkDelivery();
        this.apiMetrics = apiFactory.createApiMetrics(metricsEndpoint, metricsApiKey, metricsTimeoutMillis);
        this.apiDelivery = apiFactory.createApiDelivery(deliveryEndpoint, deliveryApiKey, deliveryTimeoutMillis, warmup, maxRequestInsertions, acceptGzip);
        this.performChecks = performChecks;
        this.blockingShadowTraffic = blockingShadowTraffic;
    }

    public DeliveryResponse deliver(DeliveryRequest deliveryRequest) throws DeliveryException {
        Response response;
        boolean isShadowTraffic = this.shouldSendShadowTraffic();
        if (this.performChecks) {
            List<String> validationErrors = deliveryRequest.validate(isShadowTraffic);
            for (String validationError : validationErrors) {
                LOGGER.log(Level.WARNING, "Delivery Request Validation Error", validationError);
            }
        }
        Request request = deliveryRequest.getRequest();
        this.fillInRequestFields(request);
        CohortMembership cohortMembership = this.checkCohortMembership(deliveryRequest);
        ExecutionServer execSrv = ExecutionServer.SDK;
        boolean attemptedDeliveryApi = false;
        if (deliveryRequest.isOnlyLog() || !this.shouldApplyTreatment(cohortMembership)) {
            response = this.sdkDelivery.runDelivery(deliveryRequest);
        } else {
            attemptedDeliveryApi = true;
            try {
                response = this.apiDelivery.runDelivery(deliveryRequest);
                execSrv = ExecutionServer.API;
            }
            catch (DeliveryException ex) {
                LOGGER.log(Level.WARNING, "Error calling Delivery API, falling back", ex);
                response = this.sdkDelivery.runDelivery(deliveryRequest);
            }
        }
        if (execSrv != ExecutionServer.API || cohortMembership != null) {
            this.logToMetrics(deliveryRequest, response, cohortMembership, execSrv);
        }
        if (!attemptedDeliveryApi && isShadowTraffic) {
            this.deliverShadowTraffic(deliveryRequest);
        }
        return new DeliveryResponse(response, request.getClientRequestId(), execSrv);
    }

    private void deliverShadowTraffic(DeliveryRequest deliveryRequest) {
        if (this.blockingShadowTraffic) {
            this.doDeliverShadowTraffic(deliveryRequest);
        } else {
            this.executor.execute(() -> this.doDeliverShadowTraffic(deliveryRequest));
        }
    }

    private void doDeliverShadowTraffic(DeliveryRequest deliveryRequest) {
        try {
            DeliveryRequest requestToSend = deliveryRequest.clone();
            assert (requestToSend.getRequest().getClientInfo() != null);
            requestToSend.getRequest().getClientInfo().setClientType(ClientType.PLATFORM_SERVER);
            requestToSend.getRequest().getClientInfo().setTrafficType(TrafficType.SHADOW);
            this.apiDelivery.runDelivery(requestToSend);
        }
        catch (DeliveryException | CloneNotSupportedException ex) {
            LOGGER.log(Level.WARNING, "Error calling Delivery API for shadow traffic", ex);
        }
    }

    private boolean shouldApplyTreatment(CohortMembership cohortMembership) {
        if (this.applyTreatmentChecker != null) {
            return this.applyTreatmentChecker.shouldApplyTreatment(cohortMembership);
        }
        if (cohortMembership == null) {
            return true;
        }
        return cohortMembership.getArm() == null || cohortMembership.getArm() != CohortArm.CONTROL;
    }

    private boolean shouldSendShadowTraffic() {
        return this.shadowTrafficDeliveryRate > 0.0f && this.sampler.sampleRandom(this.shadowTrafficDeliveryRate);
    }

    private CohortMembership checkCohortMembership(DeliveryRequest deliveryRequest) {
        CohortMembership experiment = deliveryRequest.getExperiment();
        if (experiment == null) {
            return null;
        }
        CohortMembership cohortMembership = new CohortMembership().arm(experiment.getArm()).cohortId(experiment.getCohortId());
        return cohortMembership;
    }

    private void logToMetrics(DeliveryRequest deliveryRequest, Response deliveryResponse, CohortMembership cohortMembership, ExecutionServer execSrv) {
        this.executor.execute(() -> {
            LogRequest logRequest = this.createLogRequest(deliveryRequest, deliveryResponse, cohortMembership, execSrv);
            try {
                this.apiMetrics.runMetricsLogging(logRequest);
            }
            catch (DeliveryException ex) {
                LOGGER.log(Level.WARNING, "Error calling Metrics API", ex);
            }
        });
    }

    private LogRequest createLogRequest(DeliveryRequest deliveryRequest, Response response, CohortMembership cohortMembershipToLog, ExecutionServer execSvr) {
        Request request = deliveryRequest.getRequest();
        LogRequest logRequest = new LogRequest().userInfo(request.getUserInfo()).clientInfo(request.getClientInfo()).platformId(request.getPlatformId()).timing(request.getTiming());
        if (execSvr != ExecutionServer.API) {
            DeliveryLog deliveryLog = new DeliveryLog().execution(new DeliveryExecution().executionServer(execSvr).serverVersion(SERVER_VERSION)).request(request).response(response);
            logRequest.addDeliveryLogItem(deliveryLog);
        }
        if (cohortMembershipToLog != null) {
            logRequest.addCohortMembershipItem(cohortMembershipToLog);
        }
        return logRequest;
    }

    private void fillInRequestFields(Request request) {
        if (request.getClientInfo() == null) {
            request.setClientInfo(new ClientInfo());
        }
        request.getClientInfo().setClientType(ClientType.PLATFORM_SERVER);
        request.getClientInfo().setTrafficType(TrafficType.PRODUCTION);
        this.ensureClientRequestId(request);
        this.ensureClientTimestamp(request);
    }

    private void ensureClientTimestamp(Request request) {
        if (request.getTiming() == null) {
            request.setTiming(new Timing());
        }
        if (request.getTiming().getClientLogTimestamp() == null) {
            request.getTiming().setClientLogTimestamp(System.currentTimeMillis());
        }
    }

    private void ensureClientRequestId(Request request) {
        if (request.getClientRequestId() == null || request.getClientRequestId().isBlank()) {
            request.setClientRequestId(UUID.randomUUID().toString());
        }
    }

    public boolean isPerformChecks() {
        return this.performChecks;
    }

    public void setPerformChecks(boolean performChecks) {
        this.performChecks = performChecks;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private String deliveryEndpoint;
        private String deliveryApiKey;
        private long deliveryTimeoutMillis;
        private String metricsEndpoint;
        private String metricsApiKey;
        private long metricsTimeoutMillis;
        private boolean warmup;
        private Executor executor;
        private ApplyTreatmentChecker applyTreatmentChecker;
        private ApiFactory apiFactory;
        private int maxRequestInsertions;
        private float shadowTrafficDeliveryRate;
        private Sampler sampler;
        private boolean performChecks;
        private boolean blockingShadowTraffic;
        private boolean acceptGzip = true;

        private Builder() {
        }

        public Builder withDeliveryEndpoint(String deliveryEndpoint) {
            this.deliveryEndpoint = deliveryEndpoint;
            return this;
        }

        public Builder withDeliveryApiKey(String deliveryApiKey) {
            this.deliveryApiKey = deliveryApiKey;
            return this;
        }

        public Builder withDeliveryTimeoutMillis(long deliveryTimeoutMillis) {
            this.deliveryTimeoutMillis = deliveryTimeoutMillis;
            return this;
        }

        public Builder withMetricsEndpoint(String metricsEndpoint) {
            this.metricsEndpoint = metricsEndpoint;
            return this;
        }

        public Builder withMetricsApiKey(String metricsApiKey) {
            this.metricsApiKey = metricsApiKey;
            return this;
        }

        public Builder withMetricsTimeoutMillis(long metricsTimoutMillis) {
            this.deliveryTimeoutMillis = metricsTimoutMillis;
            return this;
        }

        public Builder witPerformChecks(boolean performChecks) {
            this.performChecks = performChecks;
            return this;
        }

        public Builder withBlockingShadowTraffic(boolean blockingShadowTraffic) {
            this.blockingShadowTraffic = blockingShadowTraffic;
            return this;
        }

        public Builder withWarmup(boolean warmup) {
            this.warmup = warmup;
            return this;
        }

        public Builder withExecutor(Executor executor) {
            this.executor = executor;
            return this;
        }

        public Builder withApplyTreatmentChecker(ApplyTreatmentChecker applyTreatmentChecker) {
            this.applyTreatmentChecker = applyTreatmentChecker;
            return this;
        }

        public Builder withApiFactory(ApiFactory apiFactory) {
            this.apiFactory = apiFactory;
            return this;
        }

        public Builder withMaxRequestInsertions(int maxRequestInsertions) {
            this.maxRequestInsertions = maxRequestInsertions;
            return this;
        }

        public Builder withShadowTrafficDeliveryRate(float shadowTrafficDeliveryRate) {
            this.shadowTrafficDeliveryRate = shadowTrafficDeliveryRate;
            return this;
        }

        public Builder withSampler(Sampler sampler) {
            this.sampler = sampler;
            return this;
        }

        public Builder withAcceptGzip(boolean acceptGzip) {
            this.acceptGzip = acceptGzip;
            return this;
        }

        public PromotedDeliveryClient build() {
            return new PromotedDeliveryClient(this.deliveryEndpoint, this.deliveryApiKey, this.deliveryTimeoutMillis, this.metricsEndpoint, this.metricsApiKey, this.metricsTimeoutMillis, this.warmup, this.executor, this.maxRequestInsertions, this.applyTreatmentChecker, this.apiFactory, this.shadowTrafficDeliveryRate, this.sampler, this.performChecks, this.blockingShadowTraffic, this.acceptGzip);
        }
    }
}

