package com.scandocai.scandocsdk

import android.content.Context
import android.graphics.Bitmap
import com.scandocai.scandocsdk.models.AuthenticateRefreshRequest
import com.scandocai.scandocsdk.models.AuthenticateRefreshResponse
import com.scandocai.scandocsdk.models.AuthenticateRequest
import com.scandocai.scandocsdk.models.AuthenticateResponse
import com.scandocai.scandocsdk.models.ExtractionDataFieldsRequest
import com.scandocai.scandocsdk.models.ExtractionFieldDataResponse
import com.scandocai.scandocsdk.models.ExtractionRequest
import com.scandocai.scandocsdk.models.ExtractionResponse
import com.scandocai.scandocsdk.models.ExtractionSettingsRequest
import com.scandocai.scandocsdk.models.ValidationDataFieldsRequest
import com.scandocai.scandocsdk.models.ValidationRequest
import com.scandocai.scandocsdk.models.ValidationResponse
import com.scandocai.scandocsdk.utils.Result
import com.scandocai.scandocsdk.utils.encodeToBase64
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.cancel
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.onCompletion
import kotlinx.coroutines.flow.onStart
import kotlinx.coroutines.flow.shareIn
import kotlinx.coroutines.flow.single
import kotlinx.coroutines.flow.take
import kotlinx.coroutines.launch
import kotlinx.serialization.SerializationException
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import java.io.IOException
import java.net.HttpURLConnection
import java.net.URL
import java.util.UUID

class ScanDocSDK private constructor() {

    private val keyServiceBaseUrl = "https://api.scandoc.ai/ks"
    private val scanAppBaseUrl = "https://api.scandoc.ai/ss"
    private val outputEventSubject = MutableSharedFlow<ScanDocEvent>()
    private val cameraImageStreamSubject = MutableSharedFlow<Bitmap>()
    private val coroutineScope = CoroutineScope(Dispatchers.IO)
    private val jsonDecode = Json { ignoreUnknownKeys = true }
    private val jsonEncode = Json { explicitNulls = false }
    private var userKey = ""
    private var subClient = ""
    private var accessToken = ""
    private var refreshToken = ""
    private var acceptTermsAndConditions = false

    companion object {
        private val shared = ScanDocSDK()

        fun initialize(userKey: String,
                       acceptTermsAndConditions: Boolean,
                       context: Context) {
            shared.initialize(userKey, acceptTermsAndConditions, context)
        }

        val outputEvent: Flow<ScanDocEvent>
            get() = shared.outputEventPublisher

        internal suspend fun onImageFromCamera(image: Bitmap) {
            shared.cameraImageStreamSubject.emit(image)
        }
    }

    private fun initialize(userKey: String,
                           acceptTermsAndConditions: Boolean,
                           context: Context) {
        this.userKey = userKey
        this.acceptTermsAndConditions = acceptTermsAndConditions
        val subClientKey = "subClient"
        val sharedPreference = context
            .getSharedPreferences("ScanDocSDK", Context.MODE_PRIVATE)
        sharedPreference
            .getString(subClientKey, null)?.let {
            this.subClient = it
        } ?: run {
            val subClient = UUID.randomUUID().toString()
            sharedPreference.edit().putString(subClientKey, subClient).apply()
            this.subClient = subClient
        }

    }

    private val outputEventPublisher: Flow<ScanDocEvent> = outputEventSubject
        .shareIn(coroutineScope, SharingStarted.Lazily, replay = 0)
        .onStart {
            initializeNetworkingAndOutputEvents()
        }
        .catch {
            try {
                coroutineScope.cancel()
            } catch (e: Exception) {
                return@catch
            }
        }
        .onCompletion {
            try {
                coroutineScope.cancel()
            } catch (e: Exception) {
                return@onCompletion
            }
        }
    // TODO check opening and hiding fragment

    private fun initializeNetworkingAndOutputEvents() {
        coroutineScope.launch {
            while (true) {
                val authenticateResult = authenticate(
                    keyServiceBaseUrl = keyServiceBaseUrl,
                    userKey = userKey,
                    subClient = subClient
                )
                when (authenticateResult) {
                    is Result.Success -> {
                        val authenticationResponse = authenticateResult.getOrNull()
                        accessToken = authenticationResponse?.accessToken.toString()
                        refreshToken = authenticationResponse?.refreshToken.toString()
                        while (true) {
                            val images = getValidatedImages(
                                accessToken = accessToken,
                                refreshToken = refreshToken,
                                keyServiceBaseUrl = keyServiceBaseUrl,
                                outputEventSubject = outputEventSubject,
                                acceptTermsAndConditions = acceptTermsAndConditions
                            )
                            // sleep to give time for threads to send output and pause because potential network errors
                            kotlinx.coroutines.delay(500) // milliseconds
                            val extractionResult = extract(
                                images = images,
                                keyServiceBaseUrl = keyServiceBaseUrl,
                                accessToken = accessToken,
                                refreshToken = refreshToken,
                                shouldDoAuthenticationIfNeeded = true,
                                outputEventSubject = outputEventSubject,
                                acceptTermsAndConditions = acceptTermsAndConditions
                            )
                            sendOutputFromExtractionResponse(
                                result = extractionResult,
                                outputEventSubject = outputEventSubject
                            )
                        }
                    }
                    is Result.Failure -> {
                        val error = authenticateResult.errorOrNull() ?: ScanDocEventError.BadUrl
                        outputEventSubject.emit(ScanDocEvent.NetworkError(error))
                    }
                }
                kotlinx.coroutines.delay(2000)
            }
        }
    }

    // Extraction
    private suspend fun extract(
        images: List<Bitmap>,
        keyServiceBaseUrl: String,
        accessToken: String,
        refreshToken: String,
        shouldDoAuthenticationIfNeeded: Boolean,
        outputEventSubject: MutableSharedFlow<ScanDocEvent>,
        acceptTermsAndConditions: Boolean
    ): Result<ExtractionResponse, ScanDocEventError> {
        outputEventSubject.emit(ScanDocEvent.ExtractionInProgress)
        val frontImageBase64 = images.getOrNull(0)?.encodeToBase64()
        val backImageBase64 = images.getOrNull(1)?.encodeToBase64()
        kotlinx.coroutines.delay(100)
        val dataFields = ExtractionDataFieldsRequest(
            frontImage = frontImageBase64,
            frontImageType = "base64",
            frontImageCropped = false,
            backImage = backImageBase64,
            backImageType = if (backImageBase64 != null) "base64" else null,
            backImageCropped = if (backImageBase64 != null) false else null
        )
        val settings = ExtractionSettingsRequest(
            ignoreBackImage = true,
            shouldReturnDocumentImage = true,
            shouldReturnFaceIfDetected = true,
            shouldReturnSignatureIfDetected = true,
            skipDocumentsSizeCheck = true,
            skipImageSizeCheck = true,
            canStoreImages = false,
            enforceDocsSameCountryTypeSeries = false,
            caseSensitiveOutput = true,
            storeFaceImage = false,
            dontUseValidation = true
        )
        val extractionRequest = ExtractionRequest(
            acceptTermsAndConditions = acceptTermsAndConditions,
            dataFields = dataFields,
            settings = settings
        )
        val url = URL("$scanAppBaseUrl/extraction/")
        val connection = url.openConnection() as HttpURLConnection
        connection.requestMethod = "POST"
        connection.connectTimeout = 10000
        connection.readTimeout = 10000
        connection.setRequestProperty("content-type", "application/json")
        connection.setRequestProperty("accept", "application/json")
        connection.setRequestProperty("Authorization", accessToken)
        val requestBody = jsonEncode.encodeToString(extractionRequest)
        connection.doOutput = true
        connection.outputStream.use { os -> os.write(requestBody.toByteArray()) }
        val responseCode = connection.responseCode
        val responseBody = try {
            connection.inputStream.bufferedReader().use { it.readText() }
        } catch (e: Exception) {
            return Result.failure(ScanDocEventError.BadServerResponse)
        }
        if (responseCode != HttpURLConnection.HTTP_OK) {
            if (responseCode == HttpURLConnection.HTTP_UNAUTHORIZED
                && shouldDoAuthenticationIfNeeded) {
                val authenticationRefresh = authenticateRefresh(
                    keyServiceBaseUrl = keyServiceBaseUrl,
                    refreshToken = refreshToken
                ).getOrNull()

                val newAccessToken = authenticationRefresh?.accessToken ?: return Result.failure(ScanDocEventError.UnableToAuthenticate)

                return extract(
                    images = images,
                    keyServiceBaseUrl = keyServiceBaseUrl,
                    accessToken = newAccessToken,
                    refreshToken = refreshToken,
                    shouldDoAuthenticationIfNeeded = false,
                    outputEventSubject = outputEventSubject,
                    acceptTermsAndConditions = acceptTermsAndConditions
                )
            } else {
                return Result.failure(ScanDocEventError.BadServerResponse)
            }
        }
        val extractionResponse = try {
            jsonDecode.decodeFromString<ExtractionResponse>(responseBody)
        } catch (e: Exception) {
            return Result.failure(ScanDocEventError.CannotParseResponse)
        }

        return Result.success(extractionResponse)
    }

    private suspend fun sendOutputFromExtractionResponse(
        result: Result<ExtractionResponse, ScanDocEventError>,
        outputEventSubject: MutableSharedFlow<ScanDocEvent>
    ) {
        val event: ScanDocEvent = when (result) {
            is Result.Success -> {
                val extractionResponse = result.value

                val documentImages = extractionResponse.imageData?.documents?.mapNotNull { stringData ->
                    val data = android.util.Base64.decode(stringData, android.util.Base64.DEFAULT)
                    android.graphics.BitmapFactory.decodeByteArray(data, 0, data.size)
                }

                val faceImage: Bitmap? = extractionResponse.imageData?.faceImage?.let { faceImageString ->
                    val faceImageData = android.util.Base64.decode(faceImageString, android.util.Base64.DEFAULT)
                    android.graphics.BitmapFactory.decodeByteArray(faceImageData, 0, faceImageData.size)
                }

                val signatureImage: Bitmap? = extractionResponse.imageData?.signature?.let { signatureImageString ->
                    val signatureImageData = android.util.Base64.decode(signatureImageString, android.util.Base64.DEFAULT)
                    android.graphics.BitmapFactory.decodeByteArray(signatureImageData, 0, signatureImageData.size)
                }

                val fields: Map<ExtractedFieldType, String?> = mapOf(
                    ExtractedFieldType.Name to getValueFromFieldData(extractionResponse.data?.name),
                    ExtractedFieldType.Surname to getValueFromFieldData(extractionResponse.data?.surname),
                    ExtractedFieldType.BirthDate to getValueFromFieldData(extractionResponse.data?.birthDate),
                    ExtractedFieldType.Gender to getValueFromFieldData(extractionResponse.data?.gender),
                    ExtractedFieldType.PlaceOfBirth to getValueFromFieldData(extractionResponse.data?.placeOfBirth),
                    ExtractedFieldType.Nationality to getValueFromFieldData(extractionResponse.data?.nationality),
                    ExtractedFieldType.DocumentNumber to getValueFromFieldData(extractionResponse.data?.documentNumber),
                    ExtractedFieldType.IssuedDate to getValueFromFieldData(extractionResponse.data?.issuedDate),
                    ExtractedFieldType.ExpiryDate to getValueFromFieldData(extractionResponse.data?.expiryDate),
                    ExtractedFieldType.CountryOfIssue to getValueFromFieldData(extractionResponse.data?.countryOfIssue),
                    ExtractedFieldType.IssuingAuthority to getValueFromFieldData(extractionResponse.data?.issuingAuthority),
                    ExtractedFieldType.AddressCountry to getValueFromFieldData(extractionResponse.data?.addressCountry),
                    ExtractedFieldType.AddressZip to getValueFromFieldData(extractionResponse.data?.addressZip),
                    ExtractedFieldType.AddressCity to getValueFromFieldData(extractionResponse.data?.addressCity),
                    ExtractedFieldType.AddressCounty to getValueFromFieldData(extractionResponse.data?.addressCounty),
                    ExtractedFieldType.AddressStreet to getValueFromFieldData(extractionResponse.data?.addressStreet),
                    ExtractedFieldType.PersonalIdentificationNumber to getValueFromFieldData(extractionResponse.data?.personalIdentificationNumber),
                    ExtractedFieldType.GivenName to getValueFromFieldData(extractionResponse.data?.givenName),
                    ExtractedFieldType.FamilyName to getValueFromFieldData(extractionResponse.data?.familyName),
                    ExtractedFieldType.MothersGivenName to getValueFromFieldData(extractionResponse.data?.mothersGivenName),
                    ExtractedFieldType.MothersFamilyName to getValueFromFieldData(extractionResponse.data?.mothersFamilyName),
                    ExtractedFieldType.SecondLastName to getValueFromFieldData(extractionResponse.data?.secondLastName),
                    ExtractedFieldType.Address to getValueFromFieldData(extractionResponse.data?.address),
                    ExtractedFieldType.PlaceOfIssue to getValueFromFieldData(extractionResponse.data?.placeOfIssue),
                    ExtractedFieldType.FathersGivenName to getValueFromFieldData(extractionResponse.data?.fathersGivenName),
                    ExtractedFieldType.FathersFamilyName to getValueFromFieldData(extractionResponse.data?.fathersFamilyName)
                )

                ScanDocEvent.Extracted(documentImages, faceImage, signatureImage, fields)
            }
            is Result.Failure -> {
                ScanDocEvent.NetworkError(result.error)
            }
        }

        outputEventSubject.emit(event)
    }

    private fun getValueFromFieldData(fieldData: ExtractionFieldDataResponse?): String? {
        if (fieldData == null) {
            return null
        }

        return if (fieldData.read == true) fieldData.recommendedValue else null
    }

    // Validation
    private suspend fun getValidatedImages(
        accessToken: String,
        refreshToken: String,
        keyServiceBaseUrl: String,
        outputEventSubject: MutableSharedFlow<ScanDocEvent>,
        acceptTermsAndConditions: Boolean
    ): List<Bitmap> {
        val blurValues = mutableListOf<Double>()
        val doubleSideDocumentValidations = mutableMapOf<String, Pair<ValidationResponse, Bitmap>>()

        while (true) {
            val (image, resizedImage) = fetchImageFromStream()
            val validationResult = validate(
                resizedImage,
                keyServiceBaseUrl,
                accessToken,
                refreshToken,
                shouldDoAuthenticationIfNeeded = true,
                blurValues,
                acceptTermsAndConditions
            )

            when (validationResult) {
                is Result.Success -> {
                    val validationResponse = validationResult.value
                    val detectedBlurValue = validationResponse.detectedBlurValue
                    if (detectedBlurValue != null) {
                        blurValues.add(detectedBlurValue)
                    } else {
                        blurValues.clear()
                    }

                    outputEventSubject.emit(ScanDocEvent.ValidationInProgress(validationResponse.infoCode))

                    if (validationResponse.validated == true) {
                        when (validationResponse.infoCode.trim()) {
                            "1000" -> return listOf(image)
                            "1007" -> {
                                val validationSideResponse = validationResponse.side
                                if (validationSideResponse != null) {
                                    doubleSideDocumentValidations[validationSideResponse] = validationResponse to image
                                }

                                val filteredDocumentValidations = doubleSideDocumentValidations
                                    .filter { it.key == "FRONT" || it.key == "BACK" }

                                if (filteredDocumentValidations.any { it.key == "FRONT" } &&
                                    filteredDocumentValidations.any { it.key == "BACK" } &&
                                    filteredDocumentValidations.all { it.value.first.country == filteredDocumentValidations.values.first().first.country }) {
                                    val images = filteredDocumentValidations
                                        .map { it.value.second }

                                    return images
                                }
                            }
                        }
                    }
                }
                is Result.Failure -> outputEventSubject.emit(ScanDocEvent.NetworkError(validationResult.error))
            }
        }
    }

    private suspend fun validate(
        image: Bitmap,
        keyServiceBaseUrl: String,
        accessToken: String,
        refreshToken: String,
        shouldDoAuthenticationIfNeeded: Boolean,
        blurValues: List<Double>,
        acceptTermsAndConditions: Boolean
    ): Result<ValidationResponse, ScanDocEventError> {
        try {
            val url = URL("$scanAppBaseUrl/validation/")
            val connection = url.openConnection() as HttpURLConnection
            connection.requestMethod = "POST"
            connection.connectTimeout = 10000
            connection.readTimeout = 10000
            connection.setRequestProperty("content-type", "application/json")
            connection.setRequestProperty("Authorization", accessToken)

            val base64Image = image.encodeToBase64()
            val validationRequest = ValidationRequest(
                acceptTermsAndConditions,
                ValidationDataFieldsRequest(listOf(base64Image), blurValues)
            )
            val requestBody = jsonEncode.encodeToString(validationRequest)
            connection.doOutput = true
            connection.outputStream.use { os -> os.write(requestBody.toByteArray()) }

            val responseCode = connection.responseCode
            if (responseCode != HttpURLConnection.HTTP_OK) {
                if (responseCode == HttpURLConnection.HTTP_UNAUTHORIZED && shouldDoAuthenticationIfNeeded) {
                    val authenticationRefresh = authenticateRefresh(
                        keyServiceBaseUrl,
                        refreshToken
                    ).getOrNull()
                    val newAccessToken = authenticationRefresh?.accessToken
                    return if (newAccessToken != null) {
                        validate(
                            image,
                            keyServiceBaseUrl,
                            newAccessToken,
                            refreshToken,
                            false,
                            blurValues,
                            acceptTermsAndConditions
                        )
                    } else {
                        Result.failure(ScanDocEventError.UnableToAuthenticate)
                    }
                } else {
                    return Result.failure(ScanDocEventError.BadServerResponse)
                }
            }

            val responseData = connection.inputStream.bufferedReader().use { it.readText() }
            val validationResponse = jsonDecode.decodeFromString<ValidationResponse>(responseData)
            return Result.success(validationResponse)
        } catch (e: IOException) {
            return Result.failure(ScanDocEventError.BadServerResponse)
        } catch (e: SerializationException) {
            return Result.failure(ScanDocEventError.CannotParseResponse)
        } catch (e: Exception) {
            return Result.failure(ScanDocEventError.CannotParseResponse)
        }
    }

    private suspend fun fetchImageFromStream(): Pair<Bitmap, Bitmap> {
        return cameraImageStreamSubject
            .take(1)
            .map { image ->
                val heightInPixels = image.height.toDouble()
                val widthInPixels = image.width.toDouble()
                val resizingCoef = maxOf(heightInPixels, widthInPixels) / 384.0
                val newHeight = heightInPixels / resizingCoef
                val newWidth = widthInPixels / resizingCoef
                val resizedImage = Bitmap.createScaledBitmap(
                    image,
                    newWidth.toInt(),
                    newHeight.toInt(),
                    true
                )

                return@map Pair(image, resizedImage)
            }
            .single()
    }

    // Auth
    private fun authenticate(
        keyServiceBaseUrl: String,
        userKey: String,
        subClient: String
    ): Result<AuthenticateResponse, ScanDocEventError> {
        val url = URL("$keyServiceBaseUrl/authenticate/")
        try {
            val connection = url.openConnection() as HttpURLConnection
            connection.requestMethod = "POST"
            connection.connectTimeout = 10000
            connection.readTimeout = 10000
            connection.setRequestProperty("accept", "application/json")
            connection.setRequestProperty("content-type", "application/json")
            val authenticateRequest = AuthenticateRequest(userKey, subClient)
            val requestBody = jsonEncode.encodeToString(authenticateRequest)
            connection.doOutput = true
            connection.outputStream.use { os -> os.write(requestBody.toByteArray()) }
            val responseCode = connection.responseCode
            if (responseCode != HttpURLConnection.HTTP_OK) {
                return Result.failure(ScanDocEventError.BadServerResponse)
            }
            val inputStream = connection.inputStream
            val responseData = inputStream.bufferedReader().use { it.readText() }
            val authenticateResponse = jsonDecode.decodeFromString<AuthenticateResponse>(responseData)
            return Result.success(authenticateResponse)
        } catch (e: IOException) {
            return Result.failure(ScanDocEventError.BadServerResponse)
        } catch (e: Exception) {
            return Result.failure(ScanDocEventError.CannotParseResponse)
        }
    }
    private fun authenticateRefresh(
        keyServiceBaseUrl: String,
        refreshToken: String
    ): Result<AuthenticateRefreshResponse, ScanDocEventError> {
        val url = URL("$keyServiceBaseUrl/authenticate/refresh")
        try {
            val connection = url.openConnection() as HttpURLConnection
            connection.requestMethod = "POST"
            connection.connectTimeout = 10000
            connection.readTimeout = 10000
            connection.setRequestProperty("accept", "application/json")
            connection.setRequestProperty("content-type", "application/json")
            val authenticateRefreshRequest = AuthenticateRefreshRequest(refreshToken)
            val requestBody = jsonEncode.encodeToString(authenticateRefreshRequest)
            connection.doOutput = true
            connection.outputStream.use { os -> os.write(requestBody.toByteArray()) }
            val responseCode = connection.responseCode
            if (responseCode != HttpURLConnection.HTTP_OK) {
                return Result.failure(ScanDocEventError.BadServerResponse)
            }
            val inputStream = connection.inputStream
            val responseData = inputStream.bufferedReader().use { it.readText() }
            val authenticateRefreshResponse = jsonDecode.decodeFromString<AuthenticateRefreshResponse>(responseData)
            authenticateRefreshResponse.accessToken.let {
                this.accessToken = it
            }
            return Result.success(authenticateRefreshResponse)
        } catch (e: IOException) {
            return Result.failure(ScanDocEventError.BadServerResponse)
        } catch (e: Exception) {
            return Result.failure(ScanDocEventError.CannotParseResponse)
        }
    }
}
