package ai.systema.client.internal

import ai.systema.client.Tracker
import ai.systema.connection.internal.Connector
import ai.systema.endpoint.EndpointTracker
import ai.systema.enums.TrackerEventType
import ai.systema.helper.TrackEventBuilder
import ai.systema.model.tracker.cart.CartItem
import ai.systema.model.tracker.cart.OrderItem
import ai.systema.model.tracker.cart.PurchaseOrder
import ai.systema.model.tracker.view.ItemContainer
import ai.systema.model.tracker.wishlist.WishlistItem
import io.ktor.client.statement.HttpResponse

internal class TrackerImpl(
    private val connector: Connector,
    private val endpoint: EndpointTracker
) : Tracker {

    override suspend fun trackContainerShown(
        containers: List<ItemContainer>,
        url: String?,
        referrer: String?
    ): HttpResponse {
        val event = TrackEventBuilder(connector)
            .setContainers(containers)
            .setReferrer(referrer)
            .setUrl(url)
            .buildContainerShownEvent()

        return endpoint.sendContainerShownEvent(event)
    }

    override suspend fun trackPageViewed(
        productId: String,
        url: String,
        recId: String,
        referrer: String,
    ): HttpResponse {

        val event = TrackEventBuilder(connector)
            .setProductId(productId)
            .setRecId(recId)
            .setReferrer(referrer)
            .setUrl(url)
            .buildPagedViewEvent()

        return endpoint.sendPageViewEvent(event)
    }

    override suspend fun trackItemClicked(
        productId: String,
        url: String,
        recId: String,
        referrer: String,
    ): HttpResponse {

        val event = TrackEventBuilder(connector)
            .setProductId(productId)
            .setRecId(recId)
            .setReferrer(referrer)
            .setUrl(url)
            .buildItemClickEvent()

        return endpoint.sendItemClickEvent(event)
    }


    override suspend fun trackItemAcquired(items: List<CartItem>, url: String, referrer: String?): HttpResponse {
        val event = TrackEventBuilder(connector)
            .setCartItems(items)
            .setReferrer(referrer)
            .setUrl(url)
            .buildCartItemAcquired()

        return endpoint.sendCartItemAcquiredEvent(event)
    }

    override suspend fun trackItemRelinquished(item: CartItem, url: String, referrer: String?): HttpResponse {
        val event = TrackEventBuilder(connector)
            .setCartItem(item)
            .setReferrer(referrer)
            .setUrl(url)
            .buildCartItemRelinquished()

        return endpoint.sendCartItemRelinquishedEvent(event)
    }

    override suspend fun trackAcquisitionComplete(order: PurchaseOrder, url: String, referrer: String?): HttpResponse {
        val event = TrackEventBuilder(connector)
            .setPurchaseOrder(order)
            .setReferrer(referrer)
            .setUrl(url)
            .buildCartAcquisitionComplete()

        return endpoint.sendCartItemAcquisitionCompleteEvent(event)
    }

    override suspend fun trackWishlistAcquired(items: List<WishlistItem>, url: String, referrer: String?): HttpResponse {
        val event = TrackEventBuilder(connector)
            .setWishlistItems(items)
            .setReferrer(referrer)
            .setUrl(url)
            .buildWishlistItemAcquired()

        return endpoint.sendWishlistItemAcquiredEvent(event)
    }

    override suspend fun trackWishlistRelinquished(item: WishlistItem, url: String, referrer: String?): HttpResponse {
        val event = TrackEventBuilder(connector)
            .setWishlistItem(item)
            .setReferrer(referrer)
            .setUrl(url)
            .buildWishlistItemRelinquished()

        return endpoint.sendWishlistItemRelinquishedEvent(event)
    }


}