package ai.systema.endpoint.internal

import ai.systema.connection.RequestOptions
import ai.systema.connection.internal.Connector
import ai.systema.endpoint.*
import ai.systema.enums.CallType
import ai.systema.enums.EndpointType
import ai.systema.model.request.RecommendationRequest
import ai.systema.model.response.RecommendationResponse
import ai.systema.serialize.internal.JsonNoDefaults
import io.ktor.http.*

internal class EndpointRecommendationImpl(
    private val connector: Connector,
) : EndpointRecommendation {

    override suspend fun getRelated(
        payload: RecommendationRequest,
        requestOptions: RequestOptions?
    ): RecommendationResponse {
        if (payload.id == null || payload.id.isBlank()) {
            throw IllegalArgumentException("'id' field cannot be blank")
        }

        return this.sendRequest(payload, RouteProductRelated, requestOptions)
    }

    override suspend fun getComplementary(
        payload: RecommendationRequest,
        requestOptions: RequestOptions?
    ): RecommendationResponse {
        if (payload.id == null || payload.id.isBlank()) {
            throw IllegalArgumentException("'id' field cannot be blank")
        }

        return this.sendRequest(payload, RouteProductComplementary, requestOptions)
    }

    override suspend fun getSimilar(
        payload: RecommendationRequest,
        requestOptions: RequestOptions?
    ): RecommendationResponse {
        if (payload.id == null || payload.id.isBlank()) {
            throw IllegalArgumentException("'id' field cannot be blank")
        }

        return this.sendRequest(payload, RouteProductSimilar, requestOptions)
    }

    override suspend fun getCategoryTrending(
        payload: RecommendationRequest,
        requestOptions: RequestOptions?
    ): RecommendationResponse {
        if (payload.category == null || payload.category.isEmpty()) {
            throw IllegalArgumentException("'category' field cannot be empty")
        }

        if (payload.id != null) {
            throw IllegalArgumentException("'id' field should be null")
        }

        return this.sendRequest(payload, RouteCategoryTrending, requestOptions)
    }

    override suspend fun getCategoryPopular(
        payload: RecommendationRequest,
        requestOptions: RequestOptions?
    ): RecommendationResponse {
        if (payload.category == null || payload.category.isEmpty()) {
            throw IllegalArgumentException("'category' field cannot be empty")
        }

        if (payload.id != null) {
            throw IllegalArgumentException("'id' field should be null")
        }

        return this.sendRequest(payload, RouteCategoryPopular, requestOptions)
    }

    override suspend fun getTrending(
        payload: RecommendationRequest,
        requestOptions: RequestOptions?
    ): RecommendationResponse {
        if (payload.id != null || payload.category != null) {
            throw IllegalArgumentException("'category' or 'id' field should be null")
        }

        return this.sendRequest(payload, RouteTrending, requestOptions)
    }

    override suspend fun getPopular(
        payload: RecommendationRequest,
        requestOptions: RequestOptions?
    ): RecommendationResponse {
        if (payload.id != null || payload.category != null) {
            throw IllegalArgumentException("'category' or 'id' field should be null")
        }

        return this.sendRequest(payload, RoutePopular, requestOptions)
    }

    private suspend fun sendRequest(
        payload: RecommendationRequest,
        path: String,
        requestOptions: RequestOptions?
    ): RecommendationResponse {
        val body = JsonNoDefaults.encodeToString(RecommendationRequest.serializer(), payload)
        return connector.request(
            HttpMethod.Post,
            EndpointType.Recommend,
            CallType.Read,
            path,
            requestOptions,
            body
        )
    }
}