package ai.systema.android.listener

import ai.systema.client.SystemaAI
import android.view.View
import io.ktor.client.statement.HttpResponse
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.Dispatchers

/**
 * This listener monitors parent's scroll change events and invoke container shown on the child container if appropriate
 *
 * This is because in order to catch container shown event of the child, we actually need to monitor scroll change of
 * the parent view.
 */
public class SystemaOnParentScrollChangeListener(
    private val recContainer: View,
    systema: SystemaAI,
    tagMapping: Map<String, Int>,
    dispatcher: CoroutineDispatcher = Dispatchers.IO,
    callback: (Result<HttpResponse>) -> Unit,
) : View.OnScrollChangeListener, SystemaCompositeListener<View.OnScrollChangeListener>() {

    init {
        addListener(SystemaEventListener(systema, tagMapping, dispatcher, callback))
    }

    override fun onScrollChange(
        containerParent: View?,
        scrollX: Int,
        scrollY: Int,
        oldScrollX: Int,
        oldScrollY: Int,
    ) {
        containerParent ?: return
        for (listener in listenerList) {
            if (listener is SystemaEventListener) {
                // if parent's scrolling is changed, check if the child container is visible on screen or not
                // and trigger container shown event if required
                listener.checkAndTriggerContainShownEvent(recContainer)
            } else {
                listener.onScrollChange(
                    containerParent,
                    scrollX,
                    scrollY,
                    oldScrollX,
                    oldScrollY,
                )
            }
        }
    }
}
