package ai.systema.connection.internal

import ai.systema.configuration.Credentials
import ai.systema.configuration.RetryableHost
import ai.systema.enums.CallType
import ai.systema.enums.EndpointType
import io.ktor.http.*
import kotlinx.datetime.Clock

internal val Credentials.hosts
    get() = mapOf(
        EndpointType.DynamicConfig to dynamicConfigHosts,
        EndpointType.Recommend to recommendHosts,
        EndpointType.Search to searchHosts,
        EndpointType.Tracker to trackerHosts
    )

internal val Credentials.dynamicConfigHosts
    get() = listOf(
        RetryableHost(
            url = proxyUrls[EndpointType.DynamicConfig] ?: Url("http://discover.systema.cloud"),
            callType = CallType.Read
        ),
    )

internal val Credentials.trackerHosts
    get() = listOf(
        RetryableHost(
            url = proxyUrls[EndpointType.Tracker]
                ?: Url("http://tracker.${this.clientID}.${this.environment.value}.systema.cloud"),
            callType = CallType.Write
        ),
    )

internal val Credentials.searchHosts
    get() = listOf(
        RetryableHost(
            url = proxyUrls[EndpointType.Search]
                ?: Url("http://search.${this.clientID}.${this.environment.value}.systema.cloud"),
            callType = CallType.Read
        ),
    )

internal val Credentials.recommendHosts
    get() = listOf(
        RetryableHost(
            url = proxyUrls[EndpointType.Recommend]
                ?: Url("http://recommend.${this.clientID}.${this.environment.value}.systema.cloud"),
            callType = CallType.Read
        ),
    )

internal fun RetryableHost.reset() {
    lastUpdated = Clock.System.now().toEpochMilliseconds()
    isUp = true
    retryCount = 0
}

internal fun RetryableHost.hasTimedOut() {
    isUp = true
    lastUpdated = Clock.System.now().toEpochMilliseconds()
    retryCount += 1
}

internal fun RetryableHost.hasFailed() {
    isUp = false
    lastUpdated = Clock.System.now().toEpochMilliseconds()
}

internal fun List<RetryableHost>.expireHostsOlderThan(hostStatusExpirationDelayMS: Long) {
    forEach {
        val timeDelayExpired = Clock.System.now().toEpochMilliseconds() - it.lastUpdated
        if (timeDelayExpired > hostStatusExpirationDelayMS) {
            it.reset()
        }
    }
}

internal fun List<RetryableHost>.filterCallType(callType: CallType): List<RetryableHost> {
    return filter { it.callType == callType }
}
