package ai.systema.helper

import ai.systema.connection.internal.Connector
import ai.systema.constants.SystemaConstants
import ai.systema.enums.TrackerEventType
import ai.systema.model.tracker.cart.OrderItem
import ai.systema.model.tracker.cart.PurchaseOrder
import ai.systema.model.tracker.CurTrackEventDate
import ai.systema.model.tracker.view.ItemContainer
import ai.systema.model.tracker.TrackEventDate
import ai.systema.model.tracker.cart.CartItem
import ai.systema.model.tracker.cart.CartItemAcquiredEvent
import ai.systema.model.tracker.cart.CartItemAcquisitionCompleteEvent
import ai.systema.model.tracker.cart.CartItemRelinquishedEvent
import ai.systema.model.tracker.view.ContainerShownEvent
import ai.systema.model.tracker.view.ItemClickEvent
import ai.systema.model.tracker.view.PageViewEvent
import ai.systema.model.tracker.wishlist.WishlistItem
import ai.systema.model.tracker.wishlist.WishlistItemAcquiredEvent
import ai.systema.model.tracker.wishlist.WishlistItemRelinquishedEvent

internal class TrackEventBuilder(
    private val connector: Connector,
    private val eventDate: TrackEventDate = CurTrackEventDate,
) {

    private var productId: String? = null
    private var recId: String? = null
    private var referrer: String? = null
    private var url: String? = null
    private var containers: List<ItemContainer>? = null

    private var cartItem: CartItem? = null
    private var cartItems: List<CartItem>? = null

    private var wishlistItem: WishlistItem? = null
    private var wishlistItems: List<WishlistItem>? = null
    private var purchaseOrder: PurchaseOrder? = null

    internal fun setProductId(productId: String?): TrackEventBuilder {
        this.productId = productId
        return this
    }

    internal fun setRecId(recId: String?): TrackEventBuilder {
        this.recId = recId
        return this
    }

    internal fun setReferrer(referrer: String?): TrackEventBuilder {
        this.referrer = referrer
        return this
    }

    internal fun setUrl(url: String?): TrackEventBuilder {
        this.url = url
        return this
    }

    internal fun setContainers(containers: List<ItemContainer>?): TrackEventBuilder {
        this.containers = containers
        return this
    }

    internal fun setCartItem(item: CartItem?): TrackEventBuilder {
        this.cartItem = item
        return this
    }

    internal fun setCartItems(items: List<CartItem>?): TrackEventBuilder {
        this.cartItems = items
        return this
    }

    internal fun setWishlistItem(item: WishlistItem?): TrackEventBuilder {
        this.wishlistItem = item
        return this
    }

    internal fun setWishlistItems(items: List<WishlistItem>?): TrackEventBuilder {
        this.wishlistItems = items
        return this
    }

    internal fun setPurchaseOrder(purchaseOrder: PurchaseOrder?): TrackEventBuilder {
        this.purchaseOrder = purchaseOrder
        return this
    }

    internal suspend fun buildContainerShownEvent(): ContainerShownEvent {
        connector.refreshDynamicConfig()

        return ContainerShownEvent(
            clientId = connector.credentials.clientID.raw,
            environment = connector.credentials.environment.value,
            fingerprint = connector.clientUser.fingerprint,
            sessionId = connector.clientUser.sessionId,
            userAgent = connector.clientUser.userAgent,
            userName = connector.clientUser.userIdHash,
            sequence = connector.clientUser.nexSeq(),
            type = TrackerEventType.ContainerShown,
            referrer = referrer,
            url = url,
            eventDate = eventDate,
            containers = containers,
            version = SystemaConstants.Version,
        )
    }

    internal suspend fun buildItemClickEvent(): ItemClickEvent {
        connector.refreshDynamicConfig()

        return ItemClickEvent(
            clientId = connector.credentials.clientID.raw,
            environment = connector.credentials.environment.value,
            fingerprint = connector.clientUser.fingerprint,
            sessionId = connector.clientUser.sessionId,
            userAgent = connector.clientUser.userAgent,
            userName = connector.clientUser.userIdHash,
            sequence = connector.clientUser.nexSeq(),
            productId = productId,
            recId = recId,
            type = TrackerEventType.ItemClicked,
            referrer = referrer,
            url = url,
            eventDate = eventDate,
            version = SystemaConstants.Version,
        )
    }

    internal suspend fun buildPagedViewEvent(): PageViewEvent {
        connector.refreshDynamicConfig()

        return PageViewEvent(
            clientId = connector.credentials.clientID.raw,
            environment = connector.credentials.environment.value,
            fingerprint = connector.clientUser.fingerprint,
            sessionId = connector.clientUser.sessionId,
            userAgent = connector.clientUser.userAgent,
            userName = connector.clientUser.userIdHash,
            sequence = connector.clientUser.nexSeq(),
            productId = productId,
            recId = recId,
            type = TrackerEventType.PageView,
            referrer = referrer,
            url = url,
            eventDate = eventDate,
            version = SystemaConstants.Version,
        )
    }

    internal suspend fun buildCartItemAcquired(): CartItemAcquiredEvent {
        connector.refreshDynamicConfig()

        return CartItemAcquiredEvent(
            clientId = connector.credentials.clientID.raw,
            environment = connector.credentials.environment.value,
            fingerprint = connector.clientUser.fingerprint,
            sessionId = connector.clientUser.sessionId,
            userAgent = connector.clientUser.userAgent,
            userName = connector.clientUser.userIdHash,
            sequence = connector.clientUser.nexSeq(),
            type = TrackerEventType.AddToCart,
            referrer = referrer,
            url = url,
            eventDate = eventDate,
            items = cartItems,
            version = SystemaConstants.Version,
        )
    }

    internal suspend fun buildCartItemRelinquished(): CartItemRelinquishedEvent {
        connector.refreshDynamicConfig()

        return CartItemRelinquishedEvent(
            clientId = connector.credentials.clientID.raw,
            environment = connector.credentials.environment.value,
            fingerprint = connector.clientUser.fingerprint,
            sessionId = connector.clientUser.sessionId,
            userAgent = connector.clientUser.userAgent,
            userName = connector.clientUser.userIdHash,
            sequence = connector.clientUser.nexSeq(),
            type = TrackerEventType.RemoveFromCart,
            referrer = referrer,
            url = url,
            eventDate = eventDate,
            item = cartItem,
            version = SystemaConstants.Version,
        )
    }

    internal suspend fun buildCartAcquisitionComplete(): CartItemAcquisitionCompleteEvent{
        connector.refreshDynamicConfig()

        return CartItemAcquisitionCompleteEvent(
            clientId = connector.credentials.clientID.raw,
            environment = connector.credentials.environment.value,
            fingerprint = connector.clientUser.fingerprint,
            sessionId = connector.clientUser.sessionId,
            userAgent = connector.clientUser.userAgent,
            userName = connector.clientUser.userIdHash,
            sequence = connector.clientUser.nexSeq(),
            type = TrackerEventType.Purchase,
            referrer = referrer,
            url = url,
            eventDate = eventDate,
            order = purchaseOrder,
            version = SystemaConstants.Version,
        )
    }

    internal suspend fun buildWishlistItemAcquired(): WishlistItemAcquiredEvent {
        connector.refreshDynamicConfig()

        return WishlistItemAcquiredEvent(
            clientId = connector.credentials.clientID.raw,
            environment = connector.credentials.environment.value,
            fingerprint = connector.clientUser.fingerprint,
            sessionId = connector.clientUser.sessionId,
            userAgent = connector.clientUser.userAgent,
            userName = connector.clientUser.userIdHash,
            sequence = connector.clientUser.nexSeq(),
            type = TrackerEventType.AddToWishlist,
            referrer = referrer,
            url = url,
            eventDate = eventDate,
            items = wishlistItems,
            version = SystemaConstants.Version,
        )
    }

    internal suspend fun buildWishlistItemRelinquished(): WishlistItemRelinquishedEvent {
        connector.refreshDynamicConfig()

        return WishlistItemRelinquishedEvent(
            clientId = connector.credentials.clientID.raw,
            environment = connector.credentials.environment.value,
            fingerprint = connector.clientUser.fingerprint,
            sessionId = connector.clientUser.sessionId,
            userAgent = connector.clientUser.userAgent,
            userName = connector.clientUser.userIdHash,
            sequence = connector.clientUser.nexSeq(),
            type = TrackerEventType.RemoveFromWishlist,
            referrer = referrer,
            url = url,
            eventDate = eventDate,
            item = wishlistItem,
            version = SystemaConstants.Version,
        )
    }

}