package ai.systema.model

import ai.systema.configuration.SystemaKVStore
import ai.systema.constants.SystemaConstants
import ai.systema.helper.internal.UUID
import ai.systema.helper.logging.SystemaLoggerFactory
import ai.systema.model.request.RequestUser
import ai.systema.serialize.Keys
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.datetime.Clock
import kotlin.time.ExperimentalTime

public class ClientUser(
    public val fingerprint: String,
    public var sessionId: String = "",
    public val userIdHash: String = "",
    public val userAgent: String = SystemaConstants.UserAgentName,
) {
    private val logger = SystemaLoggerFactory.logger(ClientUser::class.toString())

    private var sequence: Int = 0
    private var sessionCreatedAt = Clock.System.now().toString()
    private val mutex = Mutex()

    public suspend fun nexSeq(): Int {
        return mutex.withLock {
            this.sequence++
        }
    }

    @OptIn(ExperimentalTime::class)
    public suspend fun refreshSession(kvStore: SystemaKVStore) {
        var sid = kvStore.read(Keys.SessionId)
        var createdAt = kvStore.read(Keys.SessionCreatedAt)
        logger.debug("Cached sessionId: $sessionId, createdAt: $createdAt")

        if (createdAt != null && sid != null) {
            val sessionCreatedAt = kotlinx.datetime.Instant.parse(createdAt)
            logger.debug("Parsed sessionCreatedAt: $sessionCreatedAt")
            if (sessionCreatedAt > Clock.System.now().minus(SystemaConstants.MaxSessionDuration)) {
                logger.debug("Using cached sessionId: $sessionId, createdAt: $createdAt, sequence: $sequence")
                this.sessionId = sid
                return
            }
        }

        // set to new
        logger.debug("Generating new sessionId...")
        this.sessionId = UUID.randomUUID()
        this.sequence = 0
        this.sessionCreatedAt = Clock.System.now().toString()

        kvStore.write(Keys.SessionId, this.sessionId)
        kvStore.write(Keys.SessionCreatedAt, this.sessionCreatedAt)
        logger.debug("Generated sessionId: ${this.sessionId}, sessionCreatedAt: ${this.sessionCreatedAt}, sequence: $sequence")
    }

    internal fun toRequestUser(): RequestUser {
        return RequestUser(fid = this.fingerprint, sid = this.sessionId, uid = this.userIdHash)
    }
}
