package ai.systema.serialize.internal

import io.ktor.http.*
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.encoding.Decoder
import kotlinx.serialization.encoding.Encoder
import kotlinx.serialization.json.*

internal val JsonDefault = Json { encodeDefaults = true }
internal val JsonNoDefaults = Json.Default
internal val JsonNonStrict = Json {
    ignoreUnknownKeys = true
    isLenient = true
    allowSpecialFloatingPointValues = true
    encodeDefaults = true
}

@OptIn(ExperimentalSerializationApi::class)
internal val JsonDebug = Json {
    prettyPrint = true
    prettyPrintIndent = "  "
    encodeDefaults = false
}

internal fun JsonObject.merge(jsonObject: JsonObject): JsonObject {
    return toMutableMap().run {
        putAll(jsonObject)
        JsonObject(this)
    }
}

internal fun JsonObject.urlEncode(): String? {
    return if (isNotEmpty()) {
        Parameters.build {
            entries.forEach { (key, element) ->
                when (element) {
                    is JsonPrimitive -> append(key, element.content)
                    else -> append(key, Json.encodeToString(JsonElement.serializer(), element))
                }
            }
        }.formUrlEncode()
    } else null
}


internal fun Decoder.asJsonInput() = (this as JsonDecoder).decodeJsonElement()
internal fun Encoder.asJsonOutput() = this as JsonEncoder

/**
 * Convenience method to get current element as [JsonObject] or null.
 */
internal val JsonElement.jsonObjectOrNull: JsonObject?
    get() = this as? JsonObject

/**
 * Convenience method to get current element as [JsonArray] or null.
 */
internal val JsonElement.jsonArrayOrNull: JsonArray?
    get() = this as? JsonArray

/**
 * Convenience method to get current element as [JsonPrimitive] or null.
 */
internal val JsonElement.jsonPrimitiveOrNull: JsonPrimitive?
    get() = this as? JsonPrimitive
