package ai.systema.android

import ai.systema.configuration.SystemaKVStore
import ai.systema.constants.SystemaConstants
import ai.systema.helper.logging.SystemaLogger
import ai.systema.model.config.SystemaCache
import android.content.Context
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.datetime.Clock
import kotlinx.datetime.Instant
import kotlinx.serialization.json.Json
import java.io.File
import kotlin.time.ExperimentalTime

public open class SystemaAndroidStorage(
    context: Context,
) : SystemaKVStore {

    private var lastFlushedAt: Instant? = null
    private val filePath = File(context.filesDir, SystemaConstants.SystemaCacheFileName)
    private var cache: SystemaCache = SystemaCache(values = mutableMapOf())
    private val cacheMutex = Mutex()

    init {
        this.load()
    }

    internal fun load() {
        SystemaLogger.debug("Loading systema settings from: ${filePath.absolutePath}")
        if (!filePath.isFile) {
            filePath.createNewFile()
            return
        }

        val str = filePath.inputStream().readBytes().toString(Charsets.UTF_8)
        if (str.isNotBlank()) {
            cache = Json.decodeFromString(SystemaCache.serializer(), str)
            SystemaLogger.debug("Loaded systema settings from: ${filePath.absolutePath}")
        }
    }

    @OptIn(ExperimentalTime::class)
    internal fun lazyFlush(now: Instant = Clock.System.now()) {
        // lazy flush
        if (this.lastFlushedAt == null ||
            this.lastFlushedAt!! < Clock.System.now().minus(SystemaConstants.MaxInMemoryCacheDuration)
        ) {
            this.flush(now)
        }
    }

    internal fun flush(now: Instant = Clock.System.now()) {
        SystemaLogger.debug("Writing systema settings to: ${filePath.absolutePath}")
        val str = Json.encodeToString(SystemaCache.serializer(), cache)
        filePath.outputStream().write(str.toByteArray())
        SystemaLogger.debug("Writing systema settings was successful to: ${filePath.absolutePath}")
        this.lastFlushedAt = now
    }

    override suspend fun read(key: String): String? {
        if (this.cache.values.containsKey(key)) {
            return this.cache.values[key] as String
        }

        return null
    }

    override suspend fun delete(key: String): String? {
        cacheMutex.withLock {
            val v = this.cache.values.remove(key)
            this.flush()

            return v
        }
    }

    override suspend fun lazyWrite(key: String, value: String) {
        cacheMutex.withLock {
            SystemaLogger.debug("Lazy writing systema settings to: ${filePath.absolutePath}")
            this.cache.values[key] = value
            this.lazyFlush()
        }
    }

    override suspend fun write(key: String, value: String) {
        cacheMutex.withLock {
            this.cache.values[key] = value
            this.flush()
        }
    }
}
