package ai.systema.android.listener

import ai.systema.client.SystemaAI
import android.view.View
import io.ktor.client.statement.HttpResponse
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.Dispatchers

/**
 * This listener monitors parent's scroll change events and invoke container shown on the child container if appropriate
 *
 * This is because in order to catch container shown event of the child, we actually need to monitor scroll change of
 * the parent view.
 */
public class SystemaOnParentScrollChangeListener(
    systema: SystemaAI,
    tagMapping: Map<String, Int>,
    dispatcher: CoroutineDispatcher = Dispatchers.IO,
    callback: (Result<HttpResponse>) -> Unit = {},
) : View.OnScrollChangeListener, SystemaCompositeListener<View.OnScrollChangeListener>() {

    private val childContainers: MutableMap<Int, View> = mutableMapOf()

    init {
        addListener(SystemaEventListener(systema, tagMapping, dispatcher, callback))
    }

    public fun addChildContainer(recContainer: View) {
        childContainers[recContainer.hashCode()] = recContainer
    }

    public fun removeChildContainer(recContainer: View) {
        childContainers.remove(recContainer.hashCode())
    }

    public fun getChildContainers(): List<View> {
        return childContainers.values.toList()
    }

    override fun onScrollChange(
        containerParent: View?,
        scrollX: Int,
        scrollY: Int,
        oldScrollX: Int,
        oldScrollY: Int,
    ) {
        containerParent ?: return
        for (listener in this.getListeners()) {
            if (listener is SystemaEventListener) {
                for (entry in childContainers.entries) {
                    // if parent's scrolling is changed, check if the child container is visible on screen or not
                    // and trigger container shown event if required
                    listener.checkAndTriggerContainShownEvent(entry.value)
                }
            } else {
                listener.onScrollChange(
                    containerParent,
                    scrollX,
                    scrollY,
                    oldScrollX,
                    oldScrollY,
                )
            }
        }
    }
}
