package ai.systema.client.internal

import ai.systema.TestConfig
import ai.systema.client.Recommender
import ai.systema.endpoint.EndpointRecommendation
import ai.systema.exception.SystemaApiException
import ai.systema.helper.RecommendationMockClient
import ai.systema.model.request.CartRecommendationRequest
import ai.systema.model.request.RecommendationRequest
import ai.systema.testSuspend
import io.mockk.coEvery
import io.mockk.mockk
import kotlin.test.Test
import kotlin.test.assertEquals
import kotlin.test.assertNotEquals

internal open class RecommenderImplTest : RecommendationMockClient() {
    @Test
    fun testGetRelated() = testSuspend {
        val connector = TestConfig.getConnector(this.initEngine())

        // check for failure
        val ep: EndpointRecommendation = mockk()
        coEvery { ep.getRelated(any(), any()) } throws SystemaApiException(IllegalArgumentException("test"))
        var recommender: Recommender = RecommenderImpl(connector, ep)
        recommender.getRelated(RecommendationRequest()).fold(
            onSuccess = {
                throw IllegalStateException("Should not succeed")
            },
            onFailure = { ex ->
                assertEquals(SystemaApiException::class, ex::class)
                assertEquals("API error occurred", ex.message)
            }
        )

        // check for success
        recommender = Recommender(connector)
        recommender.getRelated(RecommendationRequest(id = "WSH10_White")).fold(
            onSuccess = { resp ->
                assertNotEquals(0, resp.size)
                this.checkRecommendationResponse(resp)
            },
            onFailure = { ex ->
                throw IllegalStateException("Should not error", ex)
            }
        )
    }

    @Test
    fun testGetRelatedWithCallback() = testSuspend {
        val connector = TestConfig.getConnector(this.initEngine())
        val recommender = Recommender(connector)
        val payload = RecommendationRequest(
            id = "WSH10_White"
        )

        recommender.getRelated(payload) {
            it.onSuccess { resp ->
                assertNotEquals(0, resp.results.size)
                this.checkRecommendationResponse(resp)
            }
            it.onFailure { ex ->
                throw IllegalStateException("Should not error", ex)
            }
        }
    }

    @Test
    fun testGetComplementary() = testSuspend {
        val connector = TestConfig.getConnector(this.initEngine())

        // check for failure
        val ep: EndpointRecommendation = mockk()
        coEvery { ep.getComplementary(any(), any()) } throws SystemaApiException(IllegalArgumentException("test"))
        var recommender: Recommender = RecommenderImpl(connector, ep)
        recommender.getComplementary(RecommendationRequest(id = "WSH10_White")).fold(
            onSuccess = {
                throw IllegalStateException("Should not succeed")
            },
            onFailure = { ex ->
                assertEquals(SystemaApiException::class, ex::class)
                assertEquals("API error occurred", ex.message)
            }
        )

        // check for success
        recommender = Recommender(connector)
        recommender.getComplementary(RecommendationRequest(id = "WSH10_White")).fold(
            onSuccess = {
                assertNotEquals(0, it.size)
                this.checkRecommendationResponse(it)
            },
            onFailure = { ex ->
                throw IllegalStateException("Should not error", ex)
            }
        )
    }

    @Test
    fun testGetComplementaryWithCallback() = testSuspend {
        val connector = TestConfig.getConnector(this.initEngine())
        val recommender = Recommender(connector)
        val payload = RecommendationRequest(
            id = "WSH10_White"
        )

        recommender.getComplementary(payload) {
            it.onSuccess { resp ->
                assertNotEquals(0, resp.results.size)
                this.checkRecommendationResponse(resp)
            }
            it.onFailure { ex ->
                throw IllegalStateException("Should not error", ex)
            }
        }
    }

    @Test
    fun testGetSimilar() = testSuspend {
        val connector = TestConfig.getConnector(this.initEngine())

        // check for failure
        val ep: EndpointRecommendation = mockk()
        coEvery { ep.getSimilar(any(), any()) } throws SystemaApiException(IllegalArgumentException("test"))
        var recommender: Recommender = RecommenderImpl(connector, ep)
        recommender.getSimilar(RecommendationRequest()).fold(
            onSuccess = {
                throw IllegalStateException("Should not succeed")
            },
            onFailure = { ex ->
                assertEquals(SystemaApiException::class, ex::class)
                assertEquals("API error occurred", ex.message)
            }
        )

        // check for success
        recommender = Recommender(connector)
        recommender.getSimilar(RecommendationRequest(id = "WSH10_White")).fold(
            onSuccess = { resp ->
                assertNotEquals(0, resp.size)
                this.checkRecommendationResponse(resp)
            },
            onFailure = { ex ->
                throw IllegalStateException("Should not error", ex)
            }
        )
    }

    @Test
    fun testGetSimilarWithCallback() = testSuspend {
        val connector = TestConfig.getConnector(this.initEngine())
        val recommender = Recommender(connector)
        val payload = RecommendationRequest(
            id = "WSH10_White"
        )

        recommender.getSimilar(payload) {
            it.onSuccess { resp ->
                assertNotEquals(0, resp.results.size)
                this.checkRecommendationResponse(resp)
            }
            it.onFailure { ex ->
                throw IllegalStateException("Should not error", ex)
            }
        }
    }

    @Test
    fun testGetCategoryTrending() = testSuspend {
        val connector = TestConfig.getConnector(this.initEngine())

        // check for failure
        val ep: EndpointRecommendation = mockk()
        coEvery { ep.getCategoryTrending(any(), any()) } throws SystemaApiException(IllegalArgumentException("test"))
        var recommender: Recommender = RecommenderImpl(connector, ep)
        recommender.getCategoryTrending(RecommendationRequest(category = listOf("Men"))).fold(
            onSuccess = {
                throw IllegalStateException("Should not succeed")
            },
            onFailure = { ex ->
                assertEquals(SystemaApiException::class, ex::class)
                assertEquals("API error occurred", ex.message)
            }
        )

        // check for success
        recommender = Recommender(connector)
        recommender.getCategoryTrending(RecommendationRequest(category = listOf("Men"))).fold(
            onSuccess = { resp ->
                assertNotEquals(0, resp.size)
                this.checkRecommendationResponse(resp)
            },
            onFailure = { ex ->
                throw IllegalStateException("Should not error", ex)
            }
        )
    }

    @Test
    fun testGetCategoryTrendingWithCallback() = testSuspend {
        val connector = TestConfig.getConnector(this.initEngine())
        val recommender = Recommender(connector)

        val payload = RecommendationRequest(
            category = listOf("Men"),
        )

        recommender.getCategoryTrending(payload) {
            it.onSuccess { resp ->
                assertNotEquals(0, resp.results.size)
                this.checkRecommendationResponse(resp)
            }
            it.onFailure { ex ->
                throw IllegalStateException("Should not error", ex)
            }
        }
    }

    @Test
    fun testGetCategoryPopular() = testSuspend {
        val connector = TestConfig.getConnector(this.initEngine())

        // check for failure
        val ep: EndpointRecommendation = mockk()
        coEvery { ep.getCategoryPopular(any(), any()) } throws SystemaApiException(IllegalArgumentException("test"))
        var recommender: Recommender = RecommenderImpl(connector, ep)
        recommender.getCategoryPopular(RecommendationRequest(category = listOf("Men"))).fold(
            onSuccess = {
                throw IllegalStateException("Should not succeed")
            },
            onFailure = { ex ->
                assertEquals(SystemaApiException::class, ex::class)
                assertEquals("API error occurred", ex.message)
            }
        )

        // check for success
        recommender = Recommender(connector)
        recommender.getCategoryPopular(RecommendationRequest(category = listOf("Men"))).fold(
            onSuccess = {
                assertNotEquals(0, it.size)
                this.checkRecommendationResponse(it)
            },
            onFailure = { ex ->
                throw IllegalStateException("Should not error", ex)
            }
        )
    }

    @Test
    fun testGetCategoryPopularWithCallback() = testSuspend {
        val connector = TestConfig.getConnector(this.initEngine())
        val recommender = Recommender(connector)

        val payload = RecommendationRequest(
            category = listOf("Men"),
        )

        recommender.getCategoryPopular(payload) {
            it.onSuccess { resp ->
                assertNotEquals(0, resp.results.size)
                this.checkRecommendationResponse(resp)
            }
            it.onFailure { ex ->
                throw IllegalStateException("Should not error", ex)
            }
        }
    }

    @Test
    fun testGetTrending() = testSuspend {
        val connector = TestConfig.getConnector(this.initEngine())

        // check for failure
        val ep: EndpointRecommendation = mockk()
        coEvery { ep.getTrending(any(), any()) } throws SystemaApiException(IllegalArgumentException("test"))
        var recommender: Recommender = RecommenderImpl(connector, ep)
        recommender.getTrending(RecommendationRequest()).fold(
            onSuccess = {
                throw IllegalStateException("Should not succeed")
            },
            onFailure = { ex ->
                assertEquals(SystemaApiException::class, ex::class)
                assertEquals("API error occurred", ex.message)
            }
        )

        // check for success
        recommender = Recommender(connector)
        recommender.getTrending(RecommendationRequest()).fold(
            onSuccess = {
                assertNotEquals(0, it.size)
                this.checkRecommendationResponse(it)
            },
            onFailure = { ex ->
                throw IllegalStateException("Should not error", ex)
            }
        )
    }

    @Test
    fun testGetTrendingWithCallback() = testSuspend {
        val connector = TestConfig.getConnector(this.initEngine())
        val recommender = Recommender(connector)

        val payload = RecommendationRequest()

        recommender.getTrending(payload) {
            it.onSuccess { resp ->
                assertNotEquals(0, resp.results.size)
                this.checkRecommendationResponse(resp)
            }
            it.onFailure { ex ->
                throw IllegalStateException("Should not error", ex)
            }
        }
    }

    @Test
    fun testGetPopular() = testSuspend {
        val connector = TestConfig.getConnector(this.initEngine())

        // check for failure
        val ep: EndpointRecommendation = mockk()
        coEvery { ep.getPopular(any(), any()) } throws SystemaApiException(IllegalArgumentException("test"))
        var recommender: Recommender = RecommenderImpl(connector, ep)
        recommender.getPopular(RecommendationRequest()).fold(
            onSuccess = {
                throw IllegalStateException("Should not succeed")
            },
            onFailure = { ex ->
                assertEquals(SystemaApiException::class, ex::class)
                assertEquals("API error occurred", ex.message)
            }
        )

        // check for success
        recommender = Recommender(connector)
        recommender.getPopular(RecommendationRequest()).fold(
            onSuccess = {
                assertNotEquals(0, it.size)
                this.checkRecommendationResponse(it)
            },
            onFailure = { ex ->
                throw IllegalStateException("Should not error", ex)
            }
        )
    }

    @Test
    fun testGetPopularWithCallback() = testSuspend {
        val connector = TestConfig.getConnector(this.initEngine())
        val recommender = Recommender(connector)

        val payload = RecommendationRequest()

        recommender.getPopular(payload) {
            it.onSuccess { resp ->
                assertNotEquals(0, resp.results.size)
                this.checkRecommendationResponse(resp)
            }
            it.onFailure { ex ->
                throw IllegalStateException("Should not error", ex)
            }
        }
    }

    @Test
    fun testGetCartRelated() = testSuspend {
        val connector = TestConfig.getConnector(this.initEngine())

        // check for failure
        val ep: EndpointRecommendation = mockk()
        coEvery { ep.getCartRelated(any(), any()) } throws SystemaApiException(IllegalArgumentException("test"))
        var recommender: Recommender = RecommenderImpl(connector, ep)
        recommender.getCartRelated(CartRecommendationRequest(id = listOf("WSH10_White"))).fold(
            onSuccess = {
                throw IllegalStateException("Should not succeed")
            },
            onFailure = { ex ->
                assertEquals(SystemaApiException::class, ex::class)
                assertEquals("API error occurred", ex.message)
            }
        )

        // check for success
        recommender = Recommender(connector)
        recommender.getCartRelated(CartRecommendationRequest(id = listOf("WSH10_White"))).fold(
            onSuccess = {
                assertNotEquals(0, it.size)
                this.checkRecommendationResponse(it)
            },
            onFailure = { ex ->
                throw IllegalStateException("Should not error", ex)
            }
        )
    }

    @Test
    fun testGetCartRelatedWithCallback() = testSuspend {
        val connector = TestConfig.getConnector(this.initEngine())
        val recommender = Recommender(connector)

        val payload = CartRecommendationRequest(
            id = listOf("WSH10_White"),
        )

        recommender.getCartRelated(payload) {
            it.onSuccess { resp ->
                assertNotEquals(0, resp.results.size)
                this.checkRecommendationResponse(resp)
            }
            it.onFailure { ex ->
                throw IllegalStateException("Should not error", ex)
            }
        }
    }

    @Test
    fun testGetCartComplementary() = testSuspend {
        val connector = TestConfig.getConnector(this.initEngine())

        // check for failure
        val ep: EndpointRecommendation = mockk()
        coEvery { ep.getCartComplementary(any(), any()) } throws SystemaApiException(IllegalArgumentException("test"))
        var recommender: Recommender = RecommenderImpl(connector, ep)
        recommender.getCartComplementary(CartRecommendationRequest(id = listOf("WSH10_White"))).fold(
            onSuccess = {
                throw IllegalStateException("Should not succeed")
            },
            onFailure = { ex ->
                assertEquals(SystemaApiException::class, ex::class)
                assertEquals("API error occurred", ex.message)
            }
        )

        // check for success
        recommender = Recommender(connector)
        recommender.getCartComplementary(CartRecommendationRequest(id = listOf("WSH10_White"))).fold(
            onSuccess = {
                assertNotEquals(0, it.size)
                this.checkRecommendationResponse(it)
            },
            onFailure = { ex ->
                throw IllegalStateException("Should not error", ex)
            }
        )
    }

    @Test
    fun testGetCartComplementaryWithCallback() = testSuspend {
        val connector = TestConfig.getConnector(this.initEngine())
        val recommender = Recommender(connector)
        recommender.getCartComplementary(CartRecommendationRequest(id = listOf("WSH10_White"))) {
            it.onSuccess { resp ->
                assertNotEquals(0, resp.results.size)
                this.checkRecommendationResponse(resp)
            }
            it.onFailure { ex ->
                throw IllegalStateException("Should not error", ex)
            }
        }
    }
}
