package ai.systema.android

import ai.systema.testSuspend
import android.content.Context
import io.mockk.every
import io.mockk.mockk
import kotlinx.datetime.Clock
import java.io.File
import kotlin.test.Test
import kotlin.test.assertEquals
import kotlin.test.assertNotEquals
import kotlin.test.assertNull
import kotlin.test.assertTrue
import kotlin.time.Duration.Companion.seconds
import kotlin.time.ExperimentalTime

internal class SystemaAndroidStorageTest {

    private fun mockContext(): Context {
        val tmpDir = System.getProperty("java.io.tmpdir", ".")
        val fileDir = File(String.format("%s/systema-%d", tmpDir, Clock.System.now().epochSeconds))
        assertTrue { fileDir.mkdir() }

        val context: Context = mockk()
        every { context.filesDir } returns fileDir

        return context
    }

    @OptIn(ExperimentalTime::class)
    @Test
    fun lazyWrite() = testSuspend {
        val context = mockContext()
        val kvStore = SystemaAndroidStorage(context)

        val key = "time"
        val value = Clock.System.now().toString()
        assertNull(kvStore.read(key))
        kvStore.lazyWrite(key, value)
        assertEquals(value, kvStore.read(key))

        // first lazy write will write immediately to persist
        val kvStore2 = SystemaAndroidStorage(context)
        assertEquals(kvStore.read(key), kvStore2.read(key))

        // lazy write a new value, it wouldn't persist in the file immediately
        kvStore.lazyWrite(key, Clock.System.now().plus(100.seconds).toString())

        // not accessible in the file yet
        kvStore2.load()
        assertNotEquals(kvStore.read(key), kvStore2.read(key))

        // flushing and loading from the same file should sync values
        kvStore.flush()
        kvStore2.load()
        assertEquals(kvStore.read(key), kvStore.read(key))

        // write should write immediately
        kvStore.write(key, Clock.System.now().plus(100.seconds).toString())
        assertEquals(kvStore.read(key), kvStore.read(key))
    }
}
