package ai.systema.model

import ai.systema.configuration.SystemaKVStore
import ai.systema.constants.SystemaConstants
import ai.systema.constants.SystemaKeys
import ai.systema.helper.internal.UUID
import ai.systema.helper.logging.SystemaLogger
import ai.systema.model.request.RequestUser
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.datetime.Clock
import kotlinx.datetime.Instant
import kotlin.time.Duration
import kotlin.time.ExperimentalTime

public class ClientUser @OptIn(ExperimentalTime::class) constructor(
    public val fingerprint: String,
    public var sessionId: String = "",
    public var userIdHash: String = "",
    public val userAgent: String = "",
    public val maxIdleDuration: Duration = SystemaConstants.MaxSessionIdleDuration
) {

    private var sequence: Int = 0
    private var sessionLastActivityAt = Clock.System.now().toString()
    private var sessionCreatedAt = Clock.System.now().toString()
    private val mutex = Mutex()

    public suspend fun nexSeq(): Int {
        return mutex.withLock {
            this.sequence++
        }
    }

    private fun isNull(value: String?): Boolean {
        return value == null ||
            value == "null" // FIXME for iOS since it returns null as string
    }

    @OptIn(ExperimentalTime::class)
    private suspend fun resetSession(kvStore: SystemaKVStore, now: Instant = Clock.System.now()) {
        SystemaLogger.debug("Generating new sessionId...")
        this.sessionId = UUID.randomUUID()
        this.sequence = 0
        this.sessionCreatedAt = now.toString()
        this.sessionLastActivityAt = now.toString()

        kvStore.lazyWrite(SystemaKeys.SessionId, this.sessionId)
        kvStore.lazyWrite(SystemaKeys.SessionCreatedAt, this.sessionCreatedAt)
        kvStore.write(SystemaKeys.SessionLastActivityAt, this.sessionLastActivityAt)

        SystemaLogger.debug(
            "Generated sessionId: ${this.sessionId}, " +
                "sessionLastActivityAt: ${this.sessionLastActivityAt}, " +
                "sessionCreatedAt: ${this.sessionCreatedAt}, " +
                "sequence: $sequence"
        )
    }

    @OptIn(ExperimentalTime::class)
    public suspend fun refreshSession(kvStore: SystemaKVStore, now: Instant = Clock.System.now()) {
        val cachedSid = kvStore.read(SystemaKeys.SessionId)
        val cachedCreatedAt = kvStore.read(SystemaKeys.SessionCreatedAt)
        val cachedLastActivityAt = kvStore.read(SystemaKeys.SessionLastActivityAt)
        SystemaLogger.debug("Cached sessionId: $sessionId, createdAt: $cachedCreatedAt, lastActivityAt: $cachedLastActivityAt, sequence: $sequence")

        // reset sessionLastActivity
        this.sessionLastActivityAt = now.toString()
        kvStore.lazyWrite(SystemaKeys.SessionLastActivityAt, this.sessionLastActivityAt)

        // if idle period is less than the threshold, reuse existing sessionId
        if (!isNull(cachedLastActivityAt) && !isNull(cachedSid)) {
            val sessionIdleDuration = now.minus(Instant.parse(cachedLastActivityAt!!))
            if (sessionIdleDuration < maxIdleDuration) {
                SystemaLogger.debug("Re-using cached sessionId: $sessionId, sessionIdle: $sessionIdleDuration, sequence: $sequence")
                this.sessionId = cachedSid!!
                return
            }
        }

        // set to new session
        this.resetSession(kvStore, now)
    }

    /**
     * RequestUser is for request payload
     */
    internal fun toRequestUser(): RequestUser {
        return RequestUser(fid = this.fingerprint, sid = this.sessionId, uid = this.userIdHash)
    }

    /**
     * Create a snapshot of the impersonated ClientUser as SystemaUser
     */
    internal fun getSnapshot(): SystemaUser {
        return SystemaUser(
            fingerprint = this.fingerprint,
            sessionId = this.sessionId,
            userIdHash = this.userIdHash,
            userAgent = this.userAgent,
            sequence = this.sequence,
            sessionCreatedAt = Instant.parse(this.sessionCreatedAt),
            snapshotAt = Clock.System.now()
        )
    }
}
