package ai.systema.helper

import ai.systema.endpoint.SystemaRoutes
import ai.systema.model.response.SmartSuggestResponse
import io.ktor.client.engine.HttpClientEngine
import io.ktor.client.engine.mock.MockEngine
import io.ktor.client.engine.mock.MockEngineConfig
import io.ktor.client.engine.mock.respond
import io.ktor.http.ContentType
import io.ktor.http.HttpStatusCode
import io.ktor.http.headersOf
import kotlin.test.assertNotNull
import kotlin.test.assertTrue

internal open class SmartSuggestMockClient : EndpointMockClient {
    companion object {
        private val suggestRespJson: Map<String, String> = mapOf(
            SystemaRoutes.SmartSuggest to """{
                "query": "pants",
                "results": {
                    "terms": [
                        "pants"
                    ],
                    "did_you_mean": [],
                    "categories": [
                        {
                            "names": [
                                "Women",
                                "Bottoms",
                                "Pants"
                            ],
                            "link": "/women/bottoms-women/pants-women",
                            "image": "https://demo.systema.ai/media/logo/stores/1/Unreal.png"
                        },
                        {
                            "names": [
                                "Men",
                                "Bottoms",
                                "Pants"
                            ],
                            "link": "/men/bottoms-men/pants-men",
                            "image": "https://demo.systema.ai/media/logo/stores/1/Unreal.png"
                        }
                    ],
                    "products": [
                        {
                            "attributes": {
                                "size": [
                                    "32",
                                    "33",
                                    "34",
                                    "36"
                                ],
                                "color": [
                                    "Black"
                                ],
                                "options": [
                                    {
                                        "name": "Color",
                                        "position": 1,
                                        "values": [
                                            {
                                                "text": "Black",
                                                "value": "49"
                                            },
                                            {
                                                "text": "Gray",
                                                "value": "52"
                                            },
                                            {
                                                "text": "Purple",
                                                "value": "57"
                                            }
                                        ]
                                    },
                                    {
                                        "name": "Size",
                                        "position": 0,
                                        "values": [
                                            {
                                                "text": "32",
                                                "value": "175"
                                            },
                                            {
                                                "text": "33",
                                                "value": "176"
                                            },
                                            {
                                                "text": "34",
                                                "value": "177"
                                            },
                                            {
                                                "text": "36",
                                                "value": "178"
                                            }
                                        ]
                                    }
                                ]
                            },
                            "id": "MP01_Black",
                            "brand": "unreal-corp",
                            "currency": "AUD",
                            "description": "<p>Command your workout and keep your muscles limber in the Caesar Warm-Up Pant. Engineered CoolTech&trade; fabric wicks away moisture so you don't have to worry about sweat and discomfort. The drawstring-adjustable waist helps make sure your pants fit properly.</p>\n<p>&bull; Light gray heather knit straight leg pants.<br />&bull; Relaxed fit. <br />&bull; Inseam: 32\". <br />&bull; Machine wash/dry.<br />&bull; CoolTech&trade; wicking fabric.</p>",
                            "image": "https://demo.systema.ai/media/catalog/product/m/p/mp01-black_main_1.jpg",
                            "images": [
                                "https://demo.systema.ai/media/catalog/product/m/p/mp01-black_main_1.jpg"
                            ],
                            "in_stock": true,
                            "item_group_id": "MP01",
                            "link": "/caesar-warm-up-pant",
                            "price": 35.0,
                            "promotion": "",
                            "sale_price": 35.0,
                            "title": "Caesar Warm-Up Pant",
                            "tags": [],
                            "is_searchable": true,
                            "is_recommendable": true,
                            "rec_id": "tajZw17YU9TVJ2zI:v1_suggest:MP01_Black"
                        },
                        {
                            "attributes": {
                                "size": [
                                    "32",
                                    "33",
                                    "34",
                                    "36"
                                ],
                                "color": [
                                    "Blue"
                                ],
                                "options": [
                                    {
                                        "name": "Color",
                                        "position": 1,
                                        "values": [
                                            {
                                                "text": "Blue",
                                                "value": "50"
                                            },
                                            {
                                                "text": "Gray",
                                                "value": "52"
                                            },
                                            {
                                                "text": "Red",
                                                "value": "58"
                                            }
                                        ]
                                    },
                                    {
                                        "name": "Size",
                                        "position": 0,
                                        "values": [
                                            {
                                                "text": "32",
                                                "value": "175"
                                            },
                                            {
                                                "text": "33",
                                                "value": "176"
                                            },
                                            {
                                                "text": "34",
                                                "value": "177"
                                            },
                                            {
                                                "text": "36",
                                                "value": "178"
                                            }
                                        ]
                                    }
                                ]
                            },
                            "id": "MP02_Blue",
                            "brand": "unreal-corp",
                            "currency": "AUD",
                            "description": "<p>You'll love the new Viktor LumaTech&trade; Pant, with featherweight fleece fabric lining and stretchy, sweat-wicking material. It delivers toasty warmth on the sidelines or in cold-weather training, with reflective trim for a safe finish.</p>\n<p>&bull; Dark gray polyester/spandex straight leg pants.<br />&bull; Elastic waistband and internal drawstring.<br />&bull; Relaxed fit. <br />&bull; Machine wash/dry.</p>",
                            "image": "https://demo.systema.ai/media/catalog/product/m/p/mp02-blue_main_1.jpg",
                            "images": [
                                "https://demo.systema.ai/media/catalog/product/m/p/mp02-blue_main_1.jpg"
                            ],
                            "in_stock": true,
                            "item_group_id": "MP02",
                            "link": "/viktor-lumatech-trade-pant",
                            "price": 46.0,
                            "promotion": "",
                            "sale_price": 46.0,
                            "title": "Viktor LumaTech&trade; Pant",
                            "tags": [],
                            "is_searchable": true,
                            "is_recommendable": true,
                            "rec_id": "tajZw17YU9TVJ2zI:v1_suggest:MP02_Blue"
                        },
                        {
                            "attributes": {
                                "size": [
                                    "32",
                                    "33",
                                    "34",
                                    "36"
                                ],
                                "color": [
                                    "Blue"
                                ],
                                "options": [
                                    {
                                        "name": "Color",
                                        "position": 1,
                                        "values": [
                                            {
                                                "text": "Blue",
                                                "value": "50"
                                            },
                                            {
                                                "text": "Green",
                                                "value": "53"
                                            },
                                            {
                                                "text": "Red",
                                                "value": "58"
                                            }
                                        ]
                                    },
                                    {
                                        "name": "Size",
                                        "position": 0,
                                        "values": [
                                            {
                                                "text": "32",
                                                "value": "175"
                                            },
                                            {
                                                "text": "33",
                                                "value": "176"
                                            },
                                            {
                                                "text": "34",
                                                "value": "177"
                                            },
                                            {
                                                "text": "36",
                                                "value": "178"
                                            }
                                        ]
                                    }
                                ]
                            },
                            "id": "MP03_Blue",
                            "brand": "unreal-corp",
                            "currency": "AUD",
                            "description": "<p>In the cold, even the toughest guys shiver, unless they're in the Geo Insulated Jogging Pant. Lightweight and wind resistant, they block brutal wind gusts and warm you to the bone. Breathable mesh keeps them dry on the inside.</p>\n<p>&bull; Black polyester spandex pants with zipper pockets.<br />&bull; Reflective safety accents. <br />&bull; Loose fit.<br />&bull; On-seam pockets. <br />&bull; 8\" leg zips. 32\" inseam. <br />&bull; Machine wash/dry.</p>",
                            "image": "https://demo.systema.ai/media/catalog/product/m/p/mp03-blue_main_1.jpg",
                            "images": [
                                "https://demo.systema.ai/media/catalog/product/m/p/mp03-blue_main_1.jpg"
                            ],
                            "in_stock": true,
                            "item_group_id": "MP03",
                            "link": "/geo-insulated-jogging-pant",
                            "price": 51.0,
                            "promotion": "",
                            "sale_price": 51.0,
                            "title": "Geo Insulated Jogging Pant",
                            "tags": [],
                            "is_searchable": true,
                            "is_recommendable": true,
                            "rec_id": "tajZw17YU9TVJ2zI:v1_suggest:MP03_Blue"
                        }
                    ]
                },
                "total": 25,
                "result_id": "tajZw17YU9TVJ2zI:v1_suggest",
                "time": "0.17035"
            }
            """.trimIndent()
        )

        fun initMockEngine(): HttpClientEngine? {
            val config = MockEngineConfig()

            config.addHandler { request ->
                when (request.url.encodedPath) {
                    "/${SystemaRoutes.SmartSuggest}" -> {
                        respond(
                            status = HttpStatusCode.OK,
                            headers = headersOf("Content-Type", listOf(ContentType.Application.Json.toString())),
                            content = suggestRespJson[SystemaRoutes.SmartSuggest] as String
                        )
                    }
                    else -> error("Unhandled path: ${request.url.encodedPath}")
                }
            }

            return MockEngine(config)
        }
    }

    // unit test should override and return MockEngine
    // integration test should override and return null
    override fun initEngine(): HttpClientEngine? {
        return initMockEngine()
    }

    protected fun checkSmartSuggestResponse(resp: SmartSuggestResponse) {
        assertNotNull(resp.results)
        assertNotNull(resp.results.terms)
        assertNotNull(resp.results.didYouMean)
        assertNotNull(resp.results.categories)
        assertNotNull(resp.results.products)
        assertTrue { resp.results.products.isNotEmpty() }

        assertNotNull(resp.resultId)
        assertTrue { resp.resultId.isNotBlank() }

        assertNotNull(resp.time)
        assertTrue { resp.time.isNotBlank() }

        assertNotNull(resp.query)
        assertTrue { resp.query.isNotBlank() }
    }
}
