package com.particles.facebookadapter

import android.app.Activity
import android.content.Context
import android.view.View
import com.facebook.ads.Ad
import com.facebook.ads.AdError
import com.facebook.ads.AdListener
import com.facebook.ads.AdOptionsView
import com.facebook.ads.AdSettings
import com.facebook.ads.AdView
import com.facebook.ads.AudienceNetworkAds
import com.facebook.ads.BidderTokenProvider
import com.facebook.ads.InterstitialAd
import com.facebook.ads.InterstitialAdListener
import com.facebook.ads.MediaView
import com.facebook.ads.NativeAd
import com.facebook.ads.NativeAdListener
import com.facebook.biddingkit.bridge.BiddingKit
import com.particles.mes.android.MesTracker
import com.particles.mes.android.MesTrackerExt.trackAdImpression
import com.particles.mes.android.MesTrackerExt.trackAdHide
import com.particles.mes.android.MesTrackerExt.trackAdReport
import com.particles.msp.adapter.AdNetwork
import com.particles.msp.adapter.AdNetworkAdapter
import com.particles.msp.adapter.AdapterInitListener
import com.particles.msp.adapter.AdapterInitStatus
import com.particles.msp.adapter.InitializationParameters
import com.particles.msp.api.AdRequest
import com.particles.msp.api.BannerAdView
import com.particles.msp.auction.AuctionBidListener
import com.particles.msp.util.Logger
import com.particles.msp.util.Logger.info
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.prebid.mobile.rendering.bidding.data.bid.Bid
import org.prebid.mobile.rendering.bidding.data.bid.BidResponse
import kotlin.system.measureTimeMillis

class FacebookAdapter: AdNetworkAdapter() {
    private var nativeAd: NativeAd? = null
    private var adView: AdView? = null
    private var interstitialAd: InterstitialAd? = null

    override fun destroyAd() {
        nativeAd?.destroy()
        nativeAd = null
        adView?.destroy()
        adView = null
        interstitialAd?.destroy()
        interstitialAd = null
    }

    override fun initialize(
        initParams: InitializationParameters,
        adapterInitListener: AdapterInitListener,
        context: Any?
    ) {
        mesTracker = initParams.getMesHostUrl().takeIf { it.isNotEmpty() }
            ?.let { MesTracker.obtain(it) }
        CoroutineScope(Dispatchers.IO).launch {
            updateFbCCPA(initParams.isDoNotSell())
            context?.let {
                if (context is Context) {
                    AudienceNetworkAds.initialize(context)
                    BiddingKit.init(context)
                    // do not have to cache because FB will cache internally.
                    // the second getBidderToken is super fast(1ms)
                    // for loading Ads:
                    // 1. always in the background
                    // 2. always wait for google query info and facebook ready then send bid request.
                    val timeTakenFBToken = measureTimeMillis {
                        val facebookBidToken = getBidderToken(context)
                        Logger.info("FacebookAdapter. Got Facebook token: $facebookBidToken")
                    }
                    Logger.info("Time spent getting fb token: $timeTakenFBToken")
                    adapterInitListener.onComplete(AdNetwork.Facebook, AdapterInitStatus.SUCCESS, AdapterInitStatus.SUCCESS.message)
                }
            }
        }
    }

    suspend fun getBidderToken(context: Context): String {
        return BidderTokenProvider.getBidderToken(context)
    }

    override fun loadAdCreative(
        bidResponse: Any,
        auctionBidListener: AuctionBidListener,
        context: Context,
        adRequest: AdRequest,
        adListener: com.particles.msp.api.AdListener,
        bidderPlacementId: String
    ) {
        super.loadAdCreative(bidResponse, auctionBidListener, context, adRequest, adListener, bidderPlacementId)

        if (bidResponse !is BidResponse || context !is Context) {
            adListener.onError("Failed to load Facebook Ad: bidResponse or context is missing")
            return
        }
        Logger.info("loadAdCreate for facebook Ads..... : ")
        bidResponse.winningBid?.let { winnerBid ->
            val fbBid = FacebookBid(winnerBid, winnerBid.cur)
            if (winnerBid.prebid.type.equals("banner")) {
                if (adRequest.adFormat == com.particles.msp.api.AdFormat.INTERSTITIAL) {
                    loadInterstitialAd(context, fbBid, winnerBid, adListener, adRequest, auctionBidListener, bidderPlacementId)
                } else {
                    loadBannerAd(context, fbBid, winnerBid, adListener, adRequest, auctionBidListener, bidderPlacementId)
                }
            } else if (winnerBid.prebid.type.equals("native")) {
                loadNativeAd(context, fbBid, winnerBid, adListener, adRequest, auctionBidListener, bidderPlacementId)
            } else {
                adListener.onError("Load FB Ad creative: Unsupported Ad type: ${winnerBid.prebid.type}")
            }
        } ?: run {
            adListener.onError("Load FB Ad creative: No Winning bid returned")
        }
    }

    override fun prepareViewForInteraction(
        nativeAd: com.particles.msp.api.NativeAd,
        nativeAdView: Any
    ) {
        (nativeAdView as? com.particles.msp.api.NativeAdView)?.let {
            val clickableViews = mutableListOf<View>()
            nativeAdView.adTitleView?.let {
                clickableViews.add(it)
            }
            nativeAdView.adBodyView?.let {
                clickableViews.add(it)
            }
            nativeAdView.AdvertiserView?.let {
                clickableViews.add(it)
            }
            nativeAdView.AdCallToActionView?.let {
                clickableViews.add(it)
            }
            nativeAdView.AdMediaView?.let {
                clickableViews.add(it)
            }
            this.nativeAd?.registerViewForInteraction(nativeAdView,  nativeAd.mediaView as? MediaView, clickableViews)
        }
    }

    override fun sendHideAdEvent(
        reason: String,
        areaScreenshot: ByteArray?,
        fullScreenshot: ByteArray?,
    ) {
        mspAd?.let {
            mesTracker?.trackAdHide(
                adRequest,
                (bidResponse as? BidResponse)?.winningBid,
                it,
                reason,
                areaScreenshot,
                fullScreenshot,
            )
        }
    }

    override fun sendReportAdEvent(
        reason: String,
        description: String?,
        areaScreenshot: ByteArray?,
        fullScreenshot: ByteArray?,
    ) {
        mspAd?.let {
            mesTracker?.trackAdReport(
                adRequest,
                (bidResponse as? BidResponse)?.winningBid,
                it,
                reason,
                description,
                areaScreenshot,
                fullScreenshot,
            )
        }
    }

    private fun loadNativeAd(context: Context,
                             fbBid: FacebookBid,
                             winnerBid: Bid, adListener: com.particles.msp.api.AdListener,
                             adRequest: AdRequest,
                             auctionBidListener: AuctionBidListener,
                             bidderPlacementId: String) {
        val nativeAd = NativeAd(context, fbBid.placementId)
        val nativeAdListener = object : NativeAdListener {
            override fun onError(ad: Ad?, error: AdError?) {
                adListener.onError("Failed to load FB native Ad: ${error?.errorMessage}")
            }

            override fun onAdLoaded(ad: Ad?) {
                info("facebook native ad loaded ...")
                this@FacebookAdapter.nativeAd = nativeAd
                val mspNativeAd: com.particles.msp.api.NativeAd? =
                FacebookNativeAd.Builder(this@FacebookAdapter)
                    .title(nativeAd.adHeadline?:"")
                    .body(nativeAd.adBodyText?:"")
                    .advertiser(nativeAd.advertiserName?:"")
                    .callToAction(nativeAd.adCallToAction?:"")
                    .optionsView(AdOptionsView(context, nativeAd, null))
                    .mediaView(MediaView(context))
                    .build()

                mspNativeAd?.let {
                    it.adInfo["price"] = winnerBid.price
                    it.adInfo["bidder_placement_id"] = ad?.placementId ?: nativeAd.placementId
                    it.adInfo["ad_loaded_at"] = System.currentTimeMillis()
                    it.adInfo["media_view_bottom_margin"] = true
                    handleAdLoaded(it, auctionBidListener, bidderPlacementId)
                } ?: run {
                    adListener.onError("Failed to create FacebookNativeAd instance")
                }
            }

            override fun onAdClicked(ad: Ad?) {
                handleAdClicked(adListener, adRequest)
            }

            override fun onLoggingImpression(ad: Ad?) {
                mspAd?.let { mesTracker?.trackAdImpression(adRequest, winnerBid, it) }
                handleAdImpression(adListener, adRequest)
            }

            override fun onMediaDownloaded(ad: Ad?) {
            }
        }
        val configBuilder = nativeAd.buildLoadAdConfig()
            .withAdListener(nativeAdListener)
            .withBid(fbBid.payload)
        nativeAd.loadAd(configBuilder.build())
    }


    private fun loadInterstitialAd(context: Context, fbBid: FacebookBid, winnerBid: Bid, adListener: com.particles.msp.api.AdListener, adRequest: AdRequest,
                                   auctionBidListener: AuctionBidListener,
                                   bidderPlacementId: String) {
        val interstitialAd = InterstitialAd(context, fbBid.placementId)
        val interstitialAdListener = object : InterstitialAdListener {
            override fun onInterstitialDisplayed(ad: Ad) {
                mspAd?.let { mesTracker?.trackAdImpression(adRequest, winnerBid, it) }
                handleAdImpression(adListener, adRequest)
            }

            override fun onInterstitialDismissed(ad: Ad) {
                handleAdDismissed(adListener, adRequest)
            }

            override fun onError(ad: Ad, adError: AdError) {
                adListener.onError("Failed to load Facebook Interstitial Ad: ${adError.errorMessage}")
            }

            override fun onAdLoaded(ad: Ad) {
                info("successfully loaded facebook interstitial ads ...")
                this@FacebookAdapter.interstitialAd = interstitialAd
                val mspInterstitialAd = FacebookInterstitialAd(this@FacebookAdapter, interstitialAd)
                mspInterstitialAd.adInfo["price"] = winnerBid.price
                mspInterstitialAd.adInfo["bidder_placement_id"] = ad.placementId
                mspInterstitialAd.adInfo["ad_loaded_at"] = System.currentTimeMillis()
                handleAdLoaded(mspInterstitialAd, auctionBidListener, bidderPlacementId)
            }

            override fun onAdClicked(ad: Ad) {
                handleAdClicked(adListener, adRequest)
            }

            override fun onLoggingImpression(ad: Ad) {}
        }

        val configBuilder = interstitialAd.buildLoadAdConfig()
            .withAdListener(interstitialAdListener)
            .withBid(fbBid.payload)

        interstitialAd.loadAd(configBuilder.build())
    }

    private fun loadBannerAd(context: Context, fbBid: FacebookBid, winnerBid: Bid, adListener: com.particles.msp.api.AdListener, adRequest: AdRequest,
                             auctionBidListener: AuctionBidListener,
                             bidderPlacementId: String) {
        var adViewTmp: AdView? = null
        try {
            adViewTmp = AdView(context, fbBid.placementId, fbBid.payload)
        } catch (e: Exception) {
            e.printStackTrace()
        }
        val adView = adViewTmp
        adView?.let { view ->
            val facebookAdListener = object : AdListener {
                override fun onError(ad: Ad?, adError: AdError) {
                    adListener.onError("Failed to load Facebook Banner Ad: ${adError.errorMessage}")
                }

                override fun onAdLoaded(ad: Ad?) {
                    info("successfully loaded facebook banner ads ...")
                    this@FacebookAdapter.adView = view
                    val bannerAdView = BannerAdView(view, this@FacebookAdapter)
                    bannerAdView.adInfo["price"] = winnerBid.price
                    bannerAdView.adInfo["bidder_placement_id"] = ad?.placementId ?: view.placementId
                    bannerAdView.adInfo["ad_loaded_at"] = System.currentTimeMillis()
                    handleAdLoaded(bannerAdView, auctionBidListener, bidderPlacementId)
                }

                override fun onAdClicked(ad: Ad?) {
                    handleAdClicked(adListener, adRequest)
                }

                override fun onLoggingImpression(ad: Ad?) {
                    mspAd?.let { mesTracker?.trackAdImpression(adRequest, winnerBid, it) }
                    handleAdImpression(adListener, adRequest)
                }
            }

            val configBuilder = view.buildLoadAdConfig()
                .withAdListener(facebookAdListener)
                .withBid(fbBid.payload)
            view.loadAd(configBuilder.build())
        } ?: run {
            adListener.onError("Failed to create Ad view for facebook Ads")
        }
    }

    private fun updateFbCCPA(doNotSell: Boolean) {
        if (doNotSell) {
            AdSettings.setDataProcessingOptions(arrayOf("LDU"), 1, 1000)
        } else {
            AdSettings.setDataProcessingOptions(emptyArray())
        }
    }

    class FacebookInterstitialAd(facebookAdapter: FacebookAdapter, private val interstitialAd: InterstitialAd)
        : com.particles.msp.api.InterstitialAd(facebookAdapter) {
        override fun show(activity: Activity) {
            interstitialAd.show()
        }
    }

    class FacebookNativeAd(facebookAdapter: FacebookAdapter, nativeAdBuilder: com.particles.msp.api.NativeAd.Builder)
        : com.particles.msp.api.NativeAd(facebookAdapter, nativeAdBuilder) {
        @Override
        override fun prepareViewForInteraction(nativeAdView: Any) {
            adNetworkAdapter.prepareViewForInteraction(this, nativeAdView)
        }

        class Builder(private val adNetworkAdapter: FacebookAdapter) : com.particles.msp.api.NativeAd.Builder(adNetworkAdapter) {
            override fun build(): com.particles.msp.api.NativeAd {
                return FacebookNativeAd(adNetworkAdapter, this)
            }
        }
    }

    companion object {
        private var mesTracker: MesTracker? = null
    }
}