package com.particles.googleadapter

import android.app.Activity
import android.content.Context
import android.text.TextUtils
import android.view.ViewGroup
import com.google.android.gms.ads.AdLoader
import com.google.android.gms.ads.AdSize
import com.google.android.gms.ads.FullScreenContentCallback
import com.google.android.gms.ads.LoadAdError
import com.google.android.gms.ads.MobileAds
import com.google.android.gms.ads.admanager.AdManagerAdRequest
import com.google.android.gms.ads.admanager.AdManagerAdView
import com.google.android.gms.ads.admanager.AdManagerInterstitialAd
import com.google.android.gms.ads.admanager.AdManagerInterstitialAdLoadCallback
import com.google.android.gms.ads.nativead.MediaView
import com.particles.msp.adapter.AdNetwork
import com.particles.msp.adapter.AdNetworkAdapter
import com.particles.msp.adapter.AdapterInitListener
import com.particles.msp.adapter.AdapterInitStatus
import com.particles.msp.adapter.InitializationParameters
import com.particles.msp.api.AdListener
import com.particles.msp.api.AdRequest
import com.particles.msp.api.BannerAdView
import com.particles.msp.api.NativeAd
import com.particles.msp.auction.AuctionBidListener
import com.particles.msp.util.Logger.info
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.json.JSONException
import org.json.JSONObject
import org.prebid.mobile.rendering.bidding.data.bid.Bid
import org.prebid.mobile.rendering.bidding.data.bid.BidResponse
import com.google.android.gms.ads.AdListener as GoogleAdListener
import com.google.android.gms.ads.AdSize as GoogleAdSize
import com.google.android.gms.ads.nativead.NativeAd as GoogleSDKNativeAd
import com.google.android.gms.ads.nativead.NativeAdView as GoogleNativeAdView

class GoogleAdapter : AdNetworkAdapter() {
    private var adManagerAdView: AdManagerAdView? = null
    private var interstitialAd: AdManagerInterstitialAd? = null
    private var nativeAd: GoogleSDKNativeAd? = null
    private val tagPrefix = "[Adapter: Google]"

    override fun loadAdCreative(bidResponse: BidResponse?, auctionBidListener: AuctionBidListener, context: Context, adRequest: AdRequest, adListener: AdListener,
                                bidderPlacementId: String, bidderInfo: com.particles.msp.auction.BidderInfo) {
        super.loadAdCreative(bidResponse, auctionBidListener, context, adRequest, adListener, bidderPlacementId, bidderInfo)

        bidResponse!!.winningBid?.let { winnerBid ->
            if (winnerBid.prebid.type.equals("banner")) {
                if (adRequest.adFormat == com.particles.msp.api.AdFormat.INTERSTITIAL) {
                    loadInterstitialAd(context, winnerBid, adListener, adRequest, auctionBidListener, bidderPlacementId)
                } else {
                    loadBannerAd(context, winnerBid, adListener, adRequest, auctionBidListener, bidderPlacementId)
                }
            } else if (winnerBid.prebid.type.equals("native")) {
                loadNativeAd(context, winnerBid, adListener, adRequest, auctionBidListener, bidderPlacementId)
            } else {
                auctionBidListener.onError("Load FB Ad creative: Unsupported Ad type: ${winnerBid.prebid.type}")
            }
        } ?: run {
            handleAdLoadError(auctionBidListener, bidderPlacementId, "Load Google Ad creative: No Winning bid returned")
        }
    }

    private fun loadInterstitialAd(context: Context, winnerBid: Bid, adListener: AdListener, adRequest: AdRequest, auctionBidListener: AuctionBidListener,
                                   bidderPlacementId: String) {
        val builder = AdManagerAdRequest.Builder()
        winnerBid.adm?.let { builder.setAdString(it) }
        getOriginalGoogleAdUnitId(winnerBid)?.let { placementId ->
            AdManagerInterstitialAd.load(
                context,
                placementId,
                builder.build(),
                object : AdManagerInterstitialAdLoadCallback() {
                    override fun onAdLoaded(adManagerInterstitialAd: AdManagerInterstitialAd) {
                        super.onAdLoaded(adManagerInterstitialAd)
                        info("$tagPrefix successfully loaded google interstitial ads ...")
                        this@GoogleAdapter.interstitialAd = adManagerInterstitialAd
                        val mspInterstitialAd = GoogleInterstitialAd(this@GoogleAdapter, adManagerInterstitialAd)
                        adManagerInterstitialAd.fullScreenContentCallback =
                            object : FullScreenContentCallback() {
                                override fun onAdImpression() {
                                    handleAdImpression(adListener, adRequest)
                                }

                                override fun onAdClicked() {
                                    handleAdClicked(adListener, adRequest)
                                }

                                override fun onAdDismissedFullScreenContent() {
                                    handleAdDismissed(adListener, adRequest)
                                }
                            }
                        handleAdLoaded(mspInterstitialAd, auctionBidListener, bidderPlacementId, placementId, winnerBid.price)
                    }

                    override fun onAdFailedToLoad(error: LoadAdError) {
                        super.onAdFailedToLoad(error)
                        info("$tagPrefix Failed to load Google interstitial Ad: ${error.message}")
                        handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load Google interstitial Ad: ${error.message}")
                    }
                })
        }?: run {
            info("$tagPrefix Load Google Ad creative: failed to parse original placementId")
            handleAdLoadError(auctionBidListener, bidderPlacementId, "Load Google Ad creative: failed to parse original placementId")
        }
    }

    private fun loadNativeAd(context: Context, winnerBid: Bid, adListener: AdListener, adRequest: AdRequest, auctionBidListener: AuctionBidListener,
                             bidderPlacementId: String) {
        val adString = winnerBid.adm
        getOriginalGoogleAdUnitId(winnerBid)?.let { placementId ->
            val googleAdListener = object : GoogleAdListener() {
                override fun onAdImpression() {
                    super.onAdImpression()
                    handleAdImpression(adListener, adRequest)
                }

                override fun onAdClicked() {
                    super.onAdClicked()
                    handleAdClicked(adListener, adRequest)
                }

                override fun onAdFailedToLoad(adError: LoadAdError) {
                    info("$tagPrefix failed to load google native ads ...${adError.message}")
                    handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load Google native Ad: ${adError.message}")
                }
            }
            // multi-format with native and banner combined.
            val bannerAdSize = adRequest.adSize?.let {
                GoogleAdSize(it.width, it.height)
            } ?: GoogleAdSize.MEDIUM_RECTANGLE
            val loaderBuilder = AdLoader.Builder(context, placementId)
                .forNativeAd { nativeAd ->
                    info("$tagPrefix successfully loaded google native ads ...")
                    this@GoogleAdapter.nativeAd = nativeAd
                    val mspNativeAd: NativeAd =
                        GoogleNativeAd.Builder(this@GoogleAdapter)
                            .title(nativeAd.headline?:"")
                            .body(nativeAd.body?:"")
                            .advertiser(nativeAd.advertiser?:"")
                            .callToAction(nativeAd.callToAction?:"")
                            .mediaView(MediaView(context))
                            .build()

                    handleAdLoaded(mspNativeAd, auctionBidListener, bidderPlacementId, placementId, winnerBid.price)
                }
                .forAdManagerAdView({ adView ->
                    info("$tagPrefix successfully loaded google banner ads(multi-format) ...")
                    adManagerAdView = adView
                    val bannerAdView = BannerAdView(adView, this@GoogleAdapter)
                    handleAdLoaded(bannerAdView, auctionBidListener, bidderPlacementId, placementId, winnerBid.price)
                }, bannerAdSize)
                .withAdListener(googleAdListener)
            val builder = AdManagerAdRequest.Builder()
            if (!TextUtils.isEmpty(adString)) {
                builder.setAdString(adString)
            }
            loaderBuilder.build().loadAd(builder.build())
        }
    }

    private fun loadBannerAd(context: Context, winnerBid: Bid, adListener: AdListener, adRequest: AdRequest, auctionBidListener: AuctionBidListener,
                             bidderPlacementId: String) {
        val adView = AdManagerAdView(context)
        adView.setAdSizes(AdSize.BANNER)

        val adString = winnerBid.adm
        getOriginalGoogleAdUnitId(winnerBid)?.let { placementId ->
            adView.adUnitId = placementId
            adView.adListener = object : com.google.android.gms.ads.AdListener() {
                override fun onAdImpression() {
                    super.onAdImpression()
                    handleAdImpression(adListener, adRequest)
                }
                override fun onAdClicked() {
                    super.onAdClicked()
                    handleAdClicked(adListener, adRequest)
                }

                override fun onAdLoaded() {
                    info("$tagPrefix successfully loaded google banner ads ...")
                    adManagerAdView = adView
                    val bannerAdView = BannerAdView(adView, this@GoogleAdapter)
                    handleAdLoaded(bannerAdView, auctionBidListener, bidderPlacementId, placementId, winnerBid.price)
                }

                override fun onAdFailedToLoad(error: LoadAdError) {
                    info("$tagPrefix failed to load google banner ads ...${error.message}")
                    auctionBidListener.onError("Failed to load Google Banner Ad: ${error.message}")
                }
            }

            val builder = AdManagerAdRequest.Builder()
            if (!TextUtils.isEmpty(adString)) {
                builder.setAdString(adString)
            }
            try {
                adView.loadAd(builder.build())
                info("loading google banner ads ... " + winnerBid.jsonString)
            } catch (ignored: OutOfMemoryError) {
                handleAdLoadError(auctionBidListener, bidderPlacementId, "Load Google Ad creative: OutOfMemoryError when loading banner Ads")
            }
        } ?: run {
            info("$tagPrefix Load Google Ad creative: failed to parse original placementId")
            handleAdLoadError(auctionBidListener, bidderPlacementId, "Load Google Ad creative: failed to parse original placementId")
        }
    }

    private fun moveChildren(src: ViewGroup, dest: ViewGroup) {
        while (src.childCount > 0) {
            src.getChildAt(0)?.also {
                src.removeView(it)
                dest.addView(it)
            }
        }
    }

    override fun prepareViewForInteraction(nativeAd: NativeAd, nativeAdView: Any) {
        this.nativeAd?.let { googleNativeAd ->
            (nativeAdView as? com.particles.msp.api.NativeAdView)?.let { nativeAdView ->
                val googleNativeAdView = GoogleNativeAdView(nativeAdView.context).apply {
                    advertiserView = nativeAdView.AdvertiserView
                    headlineView = nativeAdView.adTitleView
                    bodyView = nativeAdView.adBodyView
                    mediaView = nativeAd.mediaView as? MediaView
                    callToActionView = nativeAdView.AdCallToActionView
                    setNativeAd(googleNativeAd)
                }
                moveChildren(nativeAdView, googleNativeAdView)
                nativeAdView.addView(googleNativeAdView)
            }
        }
    }

    override fun destroyAd() {
        adManagerAdView?.destroy()
        adManagerAdView = null
        interstitialAd?.fullScreenContentCallback = null
        interstitialAd = null
        nativeAd?.destroy()
        nativeAd = null
    }

    override fun initialize(
        initParams: InitializationParameters,
        adapterInitListener: AdapterInitListener,
        context: Context
    ) {
        super.initialize(initParams, adapterInitListener, context)
        CoroutineScope(Dispatchers.IO).launch {
            MobileAds.initialize(context) {
                adapterInitListener.onComplete(AdNetwork.Google, AdapterInitStatus.SUCCESS, AdapterInitStatus.SUCCESS.message)
                MobileAds.setAppMuted(true)
            }
        }
    }

    companion object {

        fun getOriginalGoogleAdUnitId(winner: Bid): String? {
            var adUnitId: String? = null
            try {
                val bid = JSONObject(winner.jsonString)
                val ext = bid.optJSONObject("ext")
                if (ext != null) {
                    val google = ext.optJSONObject("google")
                    if (google != null) {
                        adUnitId = google.optString("ad_unit_id")
                    }
                }
            } catch (ignored: JSONException) {}
            return adUnitId
        }
    }

    class GoogleInterstitialAd(googleAdapter: GoogleAdapter, private val interstitialAd: AdManagerInterstitialAd)
        : com.particles.msp.api.InterstitialAd(googleAdapter) {
        override fun show(activity: Activity) {
            interstitialAd.show(activity)
        }
    }

    class GoogleNativeAd(googleAdapter: GoogleAdapter, nativeAdBuilder: NativeAd.Builder)
        : NativeAd(googleAdapter, nativeAdBuilder) {
        @Override
        override fun prepareViewForInteraction(nativeAdView: Any) {
            adNetworkAdapter.prepareViewForInteraction(this, nativeAdView)
        }

        class Builder(private val adNetworkAdapter: GoogleAdapter) : NativeAd.Builder(adNetworkAdapter) {
            override fun build(): NativeAd {
                return GoogleNativeAd(adNetworkAdapter, this)
            }
        }
    }
}
