package com.particles.googleadapter

import android.os.Bundle
import android.os.Handler
import android.os.Looper
import com.google.ads.mediation.admob.AdMobAdapter
import com.google.android.gms.ads.AdFormat
import com.google.android.gms.ads.admanager.AdManagerAdRequest
import com.google.android.gms.ads.query.QueryInfo
import com.google.android.gms.ads.query.QueryInfoGenerationCallback
import com.particles.msp.adapter.GoogleQueryInfoFetcher
import com.particles.msp.adapter.GoogleQueryInfoListener
import com.particles.msp.api.MSPConstants
import com.particles.msp.util.Logger
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.prebid.mobile.rendering.sdk.ManagersResolver

class GoogleQueryInfoFetcherImp : GoogleQueryInfoFetcher {
    override fun fetch(completeListener: GoogleQueryInfoListener, adRequest: com.particles.msp.api.AdRequest) {
        val start = System.currentTimeMillis()
        val extras = Bundle()
        extras.putString("query_info_type", "requester_type_8")
        adRequest.adaptiveBannerSize?.let {
           if (it.isAnchorAdaptiveBanner) {
               // anchored adaptive banner
               extras.putInt("adaptive_banner_w", it.width);
               extras.putInt("adaptive_banner_h", it.height);
           } else if (it.isInlineAdaptiveBanner) {
               // inline adaptive banner
               extras.putInt("inlined_adaptive_banner_w", it.width);
               extras.putInt("inlined_adaptive_banner_h", it.height);
           }
        }
        val builder: AdManagerAdRequest.Builder = AdManagerAdRequest.Builder()
        builder.addNetworkExtrasBundle(AdMobAdapter::class.java, extras)

        // set neighboring content urls
        val contentUrls = mutableListOf<String>()
        contentUrls.addAll((adRequest.customParams[MSPConstants.GOOGLE_AD_MULTI_CONTENT_URLS] as? List<*>)?.filterIsInstance<String>() ?: emptyList())
        builder.setNeighboringContentUrls(contentUrls)

        val adFormat = when (adRequest.adFormat) {
            com.particles.msp.api.AdFormat.INTERSTITIAL -> AdFormat.INTERSTITIAL
            com.particles.msp.api.AdFormat.BANNER -> AdFormat.BANNER
            com.particles.msp.api.AdFormat.MULTI_FORMAT ->  {
                val googleMultFormat =  adRequest.customParams[MSPConstants.AD_REQUEST_CUSTOM_PARAM_KEY_GOOGLE_MULTI_FORMAT] as Boolean? ?: true
                if (googleMultFormat)  AdFormat.NATIVE  else  AdFormat.BANNER
            }
            else -> AdFormat.NATIVE
        }
        CoroutineScope(Dispatchers.Default).launch {
            QueryInfo.generate(ManagersResolver.getInstance().context.applicationContext,
                adFormat,
                builder.build(),
                object : QueryInfoGenerationCallback() {
                    override fun onSuccess(queryInfo: QueryInfo) {
                        Logger.verbose("Time spent to generate Google query info: ${System.currentTimeMillis() - start}ms")
                        val handler = Handler(Looper.getMainLooper())
                        handler.post { completeListener.onComplete(queryInfo.query) }
                    }

                    override fun onFailure(error: String) {
                        Logger.verbose("Time spent to generate Google query info(Failure): ${System.currentTimeMillis() - start}ms")
                    }
                })
        }
    }
}
