package com.particles.inmobiadapter

import android.content.Context
import androidx.annotation.Keep
import com.inmobi.ads.AdMetaInfo
import com.inmobi.ads.InMobiAdRequestStatus
import com.inmobi.ads.InMobiBanner
import com.inmobi.ads.InMobiInterstitial
import com.inmobi.ads.InMobiNative
import com.inmobi.ads.listeners.BannerAdEventListener
import com.inmobi.ads.listeners.InterstitialAdEventListener
import com.inmobi.ads.listeners.NativeAdEventListener
import com.inmobi.sdk.InMobiSdk
import com.inmobi.sdk.SdkInitializationListener
import com.particles.msp.adapter.AdNetwork
import com.particles.msp.adapter.AdNetworkAdapter
import com.particles.msp.adapter.AdapterInitListener
import com.particles.msp.adapter.AdapterInitStatus
import com.particles.msp.adapter.InitializationParameters
import com.particles.msp.api.MSPConstants
import com.particles.msp.auction.AuctionBidListener
import com.particles.msp.util.Logger
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import okhttp3.internal.toLongOrDefault
import org.json.JSONException
import org.json.JSONObject
import org.prebid.mobile.rendering.bidding.data.bid.BidResponse
import com.particles.msp.api.AdFormat as MspAdFormat
import com.particles.msp.api.AdListener as MspAdListener
import com.particles.msp.api.AdRequest as MspAdRequest
import com.particles.msp.api.BannerAdView as MspBannerAdView
import com.particles.msp.api.NativeAd as MspNativeAd
import com.particles.msp.api.NativeAdView as MspNativeAdView


@Keep
class InMobiAdapter : AdNetworkAdapter() {
    private var bannerAd: InMobiBanner? = null
    private var nativeAd: InMobiNative? = null
    private var interstitialAd: InMobiInterstitial? = null
    private val tagPrefix = "[Adapter: InMobi]"

    override fun destroyAd() {
        bannerAd?.destroy()
        bannerAd = null
        nativeAd?.destroy()
        nativeAd = null
        interstitialAd = null
    }

    override fun initialize(
        initParams: InitializationParameters,
        adapterInitListener: AdapterInitListener,
        context: Context
    ) {
        super.initialize(initParams, adapterInitListener, context)
        val accountId = initParams.getParameters()[MSPConstants.INIT_PARAM_KEY_INMOBI_ACCOUNT_ID] as? String
        if (accountId.isNullOrEmpty()) {
            Logger.info("$tagPrefix InMobiAdapter initialize failed: Account Id not found")
            adapterInitListener.onComplete(
                AdNetwork.InMobi,
                AdapterInitStatus.FAILURE,
                AdapterInitStatus.FAILURE.message
            )
            return
        }

        CoroutineScope(Dispatchers.IO).launch {
            if (initParams.isInTestMode()) {
                InMobiSdk.setLogLevel(InMobiSdk.LogLevel.DEBUG)
            }
            val consentObject = JSONObject()
            try {
                // Provide correct consent value to sdk which is obtained by User
                consentObject.put(InMobiSdk.IM_GDPR_CONSENT_AVAILABLE, initParams.hasUserConsent())
                // Provide 0 if GDPR is not applicable and 1 if applicable
                consentObject.put("gdpr", "0")
                // Provide user consent in IAB format
                consentObject.put(InMobiSdk.IM_GDPR_CONSENT_IAB, initParams.getConsentString())
            } catch (e: JSONException) {
                Logger.warning("$tagPrefix create InMobi consent json object failed: $e")
            }
            InMobiSdk.setIsAgeRestricted(initParams.isAgeRestrictedUser())
            InMobiSdk.init(context, accountId, consentObject,
                object : SdkInitializationListener {
                    override fun onInitializationComplete(error: Error?) {
                        if (error != null) {
                            Logger.info("$tagPrefix InMobi init failed: $error")
                            adapterInitListener.onComplete(
                                AdNetwork.InMobi,
                                AdapterInitStatus.FAILURE,
                                error.toString()
                            )
                        } else {
                            adapterInitListener.onComplete(
                                AdNetwork.InMobi,
                                AdapterInitStatus.SUCCESS,
                                AdapterInitStatus.SUCCESS.message
                            )
                        }
                    }
                })
        }
    }

    override fun loadAdCreative(
        bidResponse: BidResponse?,
        auctionBidListener: AuctionBidListener,
        context: Context,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
        bidderPlacementId: String,
        bidderInfo: com.particles.msp.auction.BidderInfo
    ) {
        super.loadAdCreative(bidResponse, auctionBidListener, context, adRequest, adListener, bidderPlacementId, bidderInfo)

        when (adRequest.adFormat) {
            MspAdFormat.BANNER -> loadBannerAd(context, bidderPlacementId, auctionBidListener, adRequest, adListener)
            MspAdFormat.NATIVE -> loadNativeAd(context, bidderPlacementId, auctionBidListener, adRequest, adListener)
            MspAdFormat.INTERSTITIAL -> loadInterstitialAd(context, bidderPlacementId, auctionBidListener, adRequest, adListener)
            else -> handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load Inmobi Ad: Unsupported Ad Format: ${adRequest.adFormat}")
        }
    }

    private fun loadBannerAd(
        context: Context,
        bidderPlacementId: String,
        auctionBidListener: AuctionBidListener,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
    ) {
        Logger.info("$tagPrefix start to load inmobi banner ads: $bidderPlacementId")

        val placementId = bidderPlacementId.toLongOrDefault(0L)
        val adEventListener = object : BannerAdEventListener() {
            override fun onAdLoadFailed(ad: InMobiBanner, status: InMobiAdRequestStatus) {
                val errorMessage = "$tagPrefix Failed to load Inmobi Banner Ad: ${status.statusCode} ${status.message}"
                Logger.info("$tagPrefix Failed to load Inmobi Banner Ad ...")
                handleAdLoadError(auctionBidListener, bidderPlacementId, errorMessage)
            }

            override fun onAdLoadSucceeded(ad: InMobiBanner, adInfo: AdMetaInfo) {
                Logger.info("$tagPrefix successfully loaded inmobi banner ads ...")
                val mspAd = MspBannerAdView(ad, this@InMobiAdapter)
                handleAdLoaded(mspAd, auctionBidListener, bidderPlacementId, bidderPlacementId, adInfo.bid)
            }

            override fun onAdImpression(ad: InMobiBanner) {
                handleAdImpression(adListener, adRequest)
            }

            override fun onAdClicked(ad: InMobiBanner, p1: MutableMap<Any, Any>?) {
                handleAdClicked(adListener, adRequest)
            }
        }

        bannerAd = InMobiBanner(context, placementId)
            .apply {
                adRequest.adSize?.let {
                    if (it.width > 0 && it.height > 0) {
                        setBannerSize(it.width, it.height)
                    }
                }
                setEnableAutoRefresh(false)
                setListener(adEventListener)
            }
            .also { it.load() }
    }

    private fun loadNativeAd(
        context: Context,
        bidderPlacementId: String,
        auctionBidListener: AuctionBidListener,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
    ) {
        Logger.info("$tagPrefix start to load inmobi native ads: $bidderPlacementId")

        val placementId = bidderPlacementId.toLongOrDefault(0L)
        val adEventListener = object : NativeAdEventListener() {
            override fun onAdLoadFailed(ad: InMobiNative, status: InMobiAdRequestStatus) {
                val errorMessage = "$tagPrefix Failed to load Inmobi native Ad: ${status.statusCode} ${status.message}"
                handleAdLoadError(auctionBidListener, bidderPlacementId, errorMessage)
            }

            override fun onAdLoadSucceeded(ad: InMobiNative, adInfo: AdMetaInfo) {
                Logger.info("$tagPrefix successfully loaded inmobi native ads ...")

                val mspAd = InMobiNativeAd.Builder(this@InMobiAdapter)
                    .title(ad.adTitle ?: "")
                    .body(ad.adDescription ?: "")
//                    .advertiser("")
                    .callToAction(ad.adCtaText ?: "")
                    .mediaView(InMobiMediaView(context).apply { setNativeAd(ad) })
                    .build()
                    .apply {
                        this.adInfo["isVideo"] = ad.isVideo == true
                    }

                handleAdLoaded(mspAd, auctionBidListener, bidderPlacementId, bidderPlacementId, adInfo.bid)
            }

            override fun onAdImpression(ad: InMobiNative) {
                handleAdImpression(adListener, adRequest)
            }

            override fun onAdClicked(ad: InMobiNative) {
                handleAdClicked(adListener, adRequest)
            }
        }

        nativeAd = InMobiNative(context, placementId, adEventListener).also { it.load() }
    }

    private fun loadInterstitialAd(
        context: Context,
        bidderPlacementId: String,
        auctionBidListener: AuctionBidListener,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
    ) {
        Logger.info("$tagPrefix start to load inmobi interstitial ads: $bidderPlacementId")

        val placementId = bidderPlacementId.toLongOrDefault(0L)
        val adEventListener = object : InterstitialAdEventListener() {
            override fun onAdLoadFailed(ad: InMobiInterstitial, status: InMobiAdRequestStatus) {
                val errorMessage = "$tagPrefix Failed to load Inmobi Interstitial Ad: ${status.statusCode} ${status.message}"
                Logger.info("$tagPrefix Failed to load Inmobi Interstitial Ad ...")
                handleAdLoadError(auctionBidListener, bidderPlacementId, errorMessage)
            }

            override fun onAdLoadSucceeded(ad: InMobiInterstitial, adInfo: AdMetaInfo) {
                Logger.info("$tagPrefix successfully loaded inmobi interstitial ads ...")
                val mspAd = InMobiInterstitialAd(this@InMobiAdapter, ad)
                handleAdLoaded(mspAd, auctionBidListener, bidderPlacementId, bidderPlacementId, adInfo.bid)
            }

            override fun onAdDisplayed(ad: InMobiInterstitial, adInfo: AdMetaInfo) {
                handleAdImpression(adListener, adRequest)
            }

            override fun onAdClicked(ad: InMobiInterstitial, params: MutableMap<Any, Any>?) {
                handleAdClicked(adListener, adRequest)
            }

            override fun onAdDismissed(ad: InMobiInterstitial) {
                handleAdDismissed(adListener, adRequest)
            }
        }

        interstitialAd = InMobiInterstitial(context, placementId, adEventListener).also { it.load() }
    }

    override fun prepareViewForInteraction(nativeAd: MspNativeAd, nativeAdView: Any) {
        if (nativeAdView !is MspNativeAdView) {
            return
        }

        val ad = this.nativeAd ?: return

        listOfNotNull(
            nativeAdView.adTitleView,
            nativeAdView.adBodyView,
            nativeAdView.AdvertiserView,
            nativeAdView.AdCallToActionView,
            nativeAdView.AdMediaView,
        ).forEach {
            it.setOnClickListener { ad.reportAdClickAndOpenLandingPage() }
        }
    }
}
