package com.particles.mobilefuseadapter

import android.content.Context
import com.mobilefuse.sdk.AdError
import com.mobilefuse.sdk.MobileFuse
import com.mobilefuse.sdk.MobileFuseBannerAd
import com.mobilefuse.sdk.MobileFuseInterstitialAd
import com.mobilefuse.sdk.MobileFuseNativeAd
import com.mobilefuse.sdk.MobileFuseSettings
import com.mobilefuse.sdk.SdkInitListener
import com.mobilefuse.sdk.privacy.MobileFusePrivacyPreferences
import com.particles.msp.adapter.AdNetwork
import com.particles.msp.adapter.AdNetworkAdapter
import com.particles.msp.adapter.AdapterInitListener
import com.particles.msp.adapter.AdapterInitStatus
import com.particles.msp.adapter.InitializationParameters
import com.particles.msp.auction.AuctionBidListener
import com.particles.msp.util.Logger
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.prebid.mobile.rendering.bidding.data.bid.BidResponse
import com.particles.msp.api.AdFormat as MspAdFormat
import com.particles.msp.api.AdListener as MspAdListener
import com.particles.msp.api.AdRequest as MspAdRequest
import com.particles.msp.api.BannerAdView as MspBannerAdView
import com.particles.msp.api.NativeAd as MspNativeAd
import com.particles.msp.api.NativeAdView as MspNativeAdView


class MobileFuseAdapter : AdNetworkAdapter() {
    private var bannerAd: MobileFuseBannerAd? = null
    private var nativeAd: MobileFuseNativeAd? = null
    private var interstitialAd: MobileFuseInterstitialAd? = null
    private val tagPrefix = "[Adapter: MobileFuse]"

    override fun destroyAd() {
        bannerAd?.destroy()
        bannerAd = null
        nativeAd?.unregisterViews()
        nativeAd = null
        interstitialAd = null
    }

    override fun initialize(
        initParams: InitializationParameters,
        adapterInitListener: AdapterInitListener,
        context: Context
    ) {
        super.initialize(initParams, adapterInitListener, context)
        CoroutineScope(Dispatchers.IO).launch {
            if (initParams.isInTestMode()) {
                MobileFuseSettings.setTestMode(true)
            }

            val privacyPrefs = MobileFusePrivacyPreferences.Builder()
                .setSubjectToCoppa(initParams.isAgeRestrictedUser())
                .build()
            MobileFuse.setPrivacyPreferences(privacyPrefs)

            MobileFuse.init(object: SdkInitListener {
                override fun onInitSuccess() {
                    adapterInitListener.onComplete(
                        AdNetwork.MobileFuse,
                        AdapterInitStatus.SUCCESS,
                        AdapterInitStatus.SUCCESS.message,
                    )
                }

                override fun onInitError() {
                    Logger.info("MobileFuse init failed")
                    adapterInitListener.onComplete(
                        AdNetwork.MobileFuse,
                        AdapterInitStatus.FAILURE,
                        AdapterInitStatus.FAILURE.message,
                    )
                }
            })
        }
    }

    override fun loadAdCreative(
        bidResponse: BidResponse?,
        auctionBidListener: AuctionBidListener,
        context: Context,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
        bidderPlacementId: String,
        bidderInfo: com.particles.msp.auction.BidderInfo
    ) {
        super.loadAdCreative(bidResponse, auctionBidListener, context, adRequest, adListener, bidderPlacementId, bidderInfo)

        when (adRequest.adFormat) {
            MspAdFormat.BANNER -> loadBannerAd(context, bidderPlacementId, auctionBidListener, adRequest, adListener)
            MspAdFormat.NATIVE -> loadNativeAd(context, bidderPlacementId, auctionBidListener, adRequest, adListener)
            MspAdFormat.INTERSTITIAL -> loadInterstitialAd(context, bidderPlacementId, auctionBidListener, adRequest, adListener)
            else -> handleAdLoadError(auctionBidListener, bidderPlacementId,"Failed to load MobileFuse Ad: Unsupported Ad Format: ${adRequest.adFormat}")
        }
    }

    private fun loadBannerAd(
        context: Context,
        bidderPlacementId: String,
        auctionBidListener: AuctionBidListener,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
    ) {
        Logger.info("$tagPrefix start to load mobilefuse banner ads: $bidderPlacementId")

        val adSize = adRequest.adSize?.let {
            when {
                it.width == 300 && it.height == 50 -> MobileFuseBannerAd.AdSize.BANNER_300x50
                it.width == 320 && it.height == 50 -> MobileFuseBannerAd.AdSize.BANNER_320x50
                it.width == 300 && it.height == 250 -> MobileFuseBannerAd.AdSize.BANNER_300x250
                it.width == 728 && it.height == 90 -> MobileFuseBannerAd.AdSize.BANNER_728x90
                else -> MobileFuseBannerAd.AdSize.BANNER_ADAPTIVE
            }
        }

        if (adSize == null) {
            auctionBidListener.onError("Failed to load MobileFuse Ad: invalid ad size.")
            return
        }

        val ad = MobileFuseBannerAd(context, bidderPlacementId, adSize)
            .apply { autorefreshEnabled = false }

        val bannerAdListener = object : MobileFuseBannerAd.Listener {
            private var isLoadResultNotified = false

            override fun onAdExpanded() {
                // Called when an 'expandable banner' placement is expanded to full-screen
            }

            override fun onAdCollapsed() {
                // Called when an 'expandable banner' placement is collapsed back from full-screen
            }

            override fun onAdLoaded() {
                if (isLoadResultNotified) return
                isLoadResultNotified = true
                Logger.info("$tagPrefix successfully loaded mobilefuse banner ads ...")
                val mspAd = MspBannerAdView(ad, this@MobileFuseAdapter)
                handleAdLoaded(mspAd, auctionBidListener, bidderPlacementId, bidderPlacementId, ad.winningBidInfo?.cpmPrice?.toDouble() ?: 0.0)
            }

            override fun onAdNotFilled() {
                if (isLoadResultNotified) return
                isLoadResultNotified = true
                Logger.info("$tagPrefix failed to load mobilefuse banner ads ... no fill")
                handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load MobileFuse banner Ad: ad not filled")
            }

            override fun onAdRendered() {
                handleAdImpression(adListener, adRequest)
            }

            override fun onAdClicked() {
                handleAdClicked(adListener, adRequest)
            }

            override fun onAdExpired() {
                // The banner ad has expired and a new ad should be loaded from the server
            }

            override fun onAdError(adError: AdError) {
                if (isLoadResultNotified) return
                isLoadResultNotified = true
                Logger.info("$tagPrefix failed to load mobilefuse banner ads ...${adError.errorCode}")
                handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load MobileFuse banner Ad: ${adError.errorMessage}")
            }
        }

        ad.setListener(bannerAdListener)
        ad.loadAd()
        bannerAd = ad
    }

    private fun loadNativeAd(
        context: Context,
        bidderPlacementId: String,
        auctionBidListener: AuctionBidListener,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
    ) {
        Logger.info("$tagPrefix start to load mobilefuse native ads: $bidderPlacementId")

        val ad = MobileFuseNativeAd(context, bidderPlacementId)
        val nativeAdListener = object : MobileFuseNativeAd.Listener {
            private var isLoadResultNotified = false

            override fun onAdClicked() {
                handleAdClicked(adListener, adRequest)
            }

            override fun onAdError(error: AdError) {
                if (isLoadResultNotified) return
                isLoadResultNotified = true
                Logger.info("$tagPrefix failed to load mobilefuse native ads ...${error.errorCode}")
                handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load MobileFuse native Ad: ${error.errorMessage}")
            }

            override fun onAdExpired() {
            }

            override fun onAdLoaded() {
                Logger.info("$tagPrefix successfully loaded mubilefuse native ads ...")

                val mspAd = MobileFuseMspNativeAd.Builder(this@MobileFuseAdapter)
                    .title(ad.getTitle(""))
                    .body(ad.getDescriptionText(""))
                    .advertiser(ad.getSponsoredText(""))
                    .callToAction(ad.getCtaButtonText(""))
                    .apply {
                        ad.getMainContentView()?.let { mediaView(it) }
                    }
                    .build()
                    .apply {
                        this.adInfo["isVideo"] = ad.hasMainVideo
                    }

                handleAdLoaded(mspAd, auctionBidListener, bidderPlacementId, bidderPlacementId, ad.winningBidInfo?.cpmPrice?.toDouble() ?: 0.0)
            }

            override fun onAdNotFilled() {
                if (isLoadResultNotified) return
                isLoadResultNotified = true
                Logger.info("$tagPrefix failed to load mobilefuse native ads ... no fill")
                handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load MobileFuse native Ad: ad not filled")
            }

            override fun onAdRendered() {
                handleAdImpression(adListener, adRequest)
            }
        }

        ad.setListener(nativeAdListener)
        ad.loadAd()
        nativeAd = ad
    }

    private fun loadInterstitialAd(
        context: Context,
        bidderPlacementId: String,
        auctionBidListener: AuctionBidListener,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
    ) {
        Logger.info("$tagPrefix start to load mobilefuse interstitial ads: $bidderPlacementId")

        val ad = MobileFuseInterstitialAd(context, bidderPlacementId)
        val interstitialAdListener = object : MobileFuseInterstitialAd.Listener {
            private var isLoadResultNotified = false

            override fun onAdLoaded() {
                Logger.info("$tagPrefix successfully loaded mobilefuse interstitial ads ...")
                val mspAd = MobileFuseMspInterstitialAd(this@MobileFuseAdapter, ad)
                handleAdLoaded(mspAd, auctionBidListener, bidderPlacementId, bidderPlacementId, ad.winningBidInfo?.cpmPrice?.toDouble() ?: 0.0)
            }

            override fun onAdNotFilled() {
                if (isLoadResultNotified) return
                isLoadResultNotified = true
                Logger.info("$tagPrefix failed to load mobilefuse interstitial ads ... no fill")
                handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load MobileFuse interstitial Ad: ad not filled")
            }

            override fun onAdRendered() {
                handleAdImpression(adListener, adRequest)
            }

            override fun onAdClicked() {
                handleAdClicked(adListener, adRequest)
            }

            override fun onAdExpired() {
            }

            override fun onAdError(error: AdError?) {
                if (isLoadResultNotified) return
                isLoadResultNotified = true
                Logger.info("$tagPrefix failed to load mobilefuse interstitial ads ...${error?.errorCode}")
                handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load MobileFuse interstitial Ad: ${error?.errorMessage}")
            }

            override fun onAdClosed() {
                handleAdDismissed(adListener, adRequest)
            }
        }
        ad.setListener(interstitialAdListener)
        ad.loadAd()
        interstitialAd = ad
    }

    override fun prepareViewForInteraction(nativeAd: MspNativeAd, nativeAdView: Any) {
        if (nativeAdView !is MspNativeAdView) {
            return
        }

        val ad = this.nativeAd ?: return

        val clickableViews = listOfNotNull(
            nativeAdView.adTitleView,
            nativeAdView.adBodyView,
            nativeAdView.AdvertiserView,
            nativeAdView.AdCallToActionView,
            nativeAdView.AdMediaView,
        )

        ad.registerViewForInteraction(nativeAdView, clickableViews)
    }
}
