package com.particles.msp.util

import com.particles.mes.android.MesTracker
import com.particles.mes.android.data.MesAdHideEvent
import com.particles.mes.android.data.MesAdImpressionEvent
import com.particles.mes.android.data.MesAdReportEvent
import com.particles.mes.android.data.MesAdRequest
import com.particles.mes.android.data.MesAdRequestExt
import com.particles.mes.android.data.MesAdResponse
import com.particles.mes.android.data.MesAdType
import com.particles.msp.api.AdRequest
import com.particles.msp.api.BannerAdView
import com.particles.msp.api.MSPAd
import com.particles.msp.api.NativeAd
import org.json.JSONObject
import org.prebid.mobile.rendering.bidding.data.bid.Bid
import org.prebid.mobile.rendering.models.openrtb.BidRequest

object MesTrackerExt {
    private fun getMockedBid(ad: MSPAd): Bid {
        val bidJsonString = """
{
  "id": "",
  "impid": "",
  "adm": "",
  "adomain": [
  ],
  "crid": "",
  "ext": {
    "prebid": {
      "meta": {
        "adaptercode": "mockSeatPlaceholder"
      }
    }
  }
}
"""
        val bidJson = JSONObject(bidJsonString)
        bidJson.put("price", ad.adInfo["price"] as? Double ?: 0.0)
        bidJson.optJSONObject("ext")?.optJSONObject("prebid")?.optJSONObject("meta")?.put("adaptercode", ad.adInfo["client_bidder"])
        return Bid.fromJSONObject(bidJson)
    }

    private fun getMesAdRequest(adRequest: AdRequest, winnerBid: Bid?, ad: MSPAd): MesAdRequest {
        return MesAdRequest(
            sentRequestAtMillis = 0L,
            bidRequest = BidRequest().apply { id = winnerBid?.requestId ?: adRequest.uuid},
            ext = MesAdRequestExt(
                source = adRequest.placementId,
                position = adRequest.customParams["position"]?.toString()?.toIntOrNull() ?: 0,
                placementId = ad.adInfo["adnetwork_placement_id"]?.toString() ?: adRequest.placementId,
                buckets = listOf(),
                adSlotId = adRequest.customParams["ad_slot_id"] as? String ?: "",
                userId = adRequest.customParams["user_id"] as? String ?: UserId.getCachedUserId()
                    ?.toString() ?: ""
            ),
            adRequest.org,
            adRequest.app,
            isOpenRtb = winnerBid != null
        )
    }

    private fun getMesAdResponse(ad: MSPAd, winnerBid: Bid?): MesAdResponse {
        return MesAdResponse(
            receivedResponseAtMillis = ad.adInfo["ad_loaded_at"] as? Long ?: 0L,
            adType = when (ad) {
                is BannerAdView -> MesAdType.AD_TYPE_DISPLAY
                is NativeAd -> MesAdType.AD_TYPE_NATIVE
                else -> MesAdType.AD_TYPE_UNSPECIFIED
            },
            bid = winnerBid ?: getMockedBid(ad),
            title = (ad as? NativeAd)?.title,
            body = (ad as? NativeAd)?.body,
            advertiser = (ad as? NativeAd)?.advertiser,
        )
    }

    fun MesTracker.trackAdImpression(adRequest: AdRequest, winnerBid: Bid?, ad: MSPAd) {
        this.trackAdImpression(
            MesAdImpressionEvent(
                adRequest = getMesAdRequest(adRequest, winnerBid, ad),
                adResponse = getMesAdResponse(ad, winnerBid)
            )
        )
    }

    fun MesTracker.trackAdHide(
        adRequest: AdRequest?, winnerBid: Bid?, ad: MSPAd,
        reason: String,
        adScreenshot: ByteArray? = null,
        fullScreenshot: ByteArray? = null
    ) {
        adRequest?.let {
            this.trackAdHide(
                MesAdHideEvent(
                    adRequest = getMesAdRequest(it, winnerBid, ad),
                    adResponse = getMesAdResponse(ad, winnerBid).copy(
                        adScreenshot = adScreenshot,
                        fullScreenshot = fullScreenshot,
                    ),
                    reason = reason
                )
            )
        }
    }

    fun MesTracker.trackAdReport(
        adRequest: AdRequest?, winnerBid: Bid?, ad: MSPAd,
        reason: String, description: String?,
        adScreenshot: ByteArray? = null,
        fullScreenshot: ByteArray? = null
    ) {
        adRequest?.let {
            this.trackAdReport(
                MesAdReportEvent(
                    adRequest = getMesAdRequest(it, winnerBid, ad),
                    adResponse = getMesAdResponse(ad, winnerBid).copy(
                        adScreenshot = adScreenshot,
                        fullScreenshot = fullScreenshot,
                    ),
                    reason = reason,
                    description = description
                )
            )
        }
    }
}