package com.particles.msp.adapter

import android.content.Context
import com.particles.mes.android.MesTracker
import com.particles.msp.AdCache
import com.particles.msp.api.AdListener
import com.particles.msp.api.AdRequest
import com.particles.msp.api.MSPAd
import com.particles.msp.api.NativeAd
import com.particles.msp.auction.AuctionBid
import com.particles.msp.auction.AuctionBidListener
import com.particles.msp.auction.BidderInfo
import com.particles.msp.util.Logger
import com.particles.msp.util.MesTrackerExt.trackAdHide
import com.particles.msp.util.MesTrackerExt.trackAdImpression
import com.particles.msp.util.MesTrackerExt.trackAdReport
import org.prebid.mobile.rendering.bidding.data.bid.BidResponse

abstract class AdNetworkAdapter {
    private var mspAd: MSPAd? = null
    private var adRequest: AdRequest? = null
    private var bidResponse: BidResponse? = null
    private var bidderInfo: BidderInfo? = null

    companion object {
        private var mesTracker: MesTracker? = null
    }

    open fun loadAdCreative(bidResponse: BidResponse?, auctionBidListener: AuctionBidListener, context: Context, adRequest: AdRequest, adListener: AdListener,
                            bidderPlacementId: String,
                            bidderInfo: BidderInfo) {
        this.adRequest = adRequest
        this.bidResponse = bidResponse
        this.bidderInfo = bidderInfo
    }

    open fun initialize(
        initParams: InitializationParameters,
        adapterInitListener: AdapterInitListener,
        context: Context) {
        if (mesTracker == null) {
            mesTracker = initParams.getMesHostUrl().takeIf { it.isNotEmpty() }
                ?.let { MesTracker.obtain(it) }
        }
    }

    abstract fun destroyAd()
    abstract fun prepareViewForInteraction(nativeAd: NativeAd, nativeAdView: Any)

    fun sendHideAdEvent(reason: String, areaScreenshot: ByteArray?, fullScreenshot: ByteArray?) {
        mspAd?.let {
            mesTracker?.trackAdHide(
                adRequest,
                bidResponse?.winningBid,
                it,
                reason,
                areaScreenshot,
                fullScreenshot,
            )
        }
    }

    fun sendReportAdEvent(reason: String, description: String?, areaScreenshot: ByteArray?, fullScreenshot: ByteArray?) {
        mspAd?.let {
            mesTracker?.trackAdReport(
                adRequest,
                bidResponse?.winningBid,
                it,
                reason,
                description,
                areaScreenshot,
                fullScreenshot,
            )
        }
    }

    fun handleAdLoadError(auctionBidListener: AuctionBidListener, bidderPlacementId: String, error: String) {
        auctionBidListener.onError(error)
    }

    fun handleAdLoaded(ad: MSPAd, auctionBidListener: AuctionBidListener, bidderPlacementId: String,
                       adNetworkPlacementId: String,
                       price: Double) {
        mspAd = ad
        mspAd?.adInfo?.set("client_bidder", bidderInfo?.name?: "")
        mspAd?.adInfo?.set("adnetwork_placement_id", adNetworkPlacementId)
        mspAd?.adInfo?.set("ad_loaded_at", System.currentTimeMillis())
        mspAd?.adInfo?.set("price", price)
        Logger.verbose("[Adapter: Base] Save Ad to cache: bidderPlacementId: $bidderPlacementId")
        AdCache.saveAd(bidderPlacementId, ad)
        auctionBidListener.onSuccess(AuctionBid("msp", bidderPlacementId, ad.adInfo["price"] as Double))
    }

    fun handleAdClicked(listener: AdListener, adRequest: AdRequest) {
        mspAd?.let {
            listener.onAdClicked(it)
        }
    }

    fun handleAdDismissed(listener: AdListener, adRequest: AdRequest) {
        mspAd?.let {
            listener.onAdDismissed(it)
        }
    }

    fun handleAdImpression(listener: AdListener, adRequest: AdRequest) {
        mspAd?.let {
            listener.onAdImpression(it)
            mesTracker?.trackAdImpression(adRequest, bidResponse?.winningBid, it)
        }
    }
}