package com.particles.msp

import android.content.Context
import com.particles.msp.adapter.AdNetwork
import com.particles.msp.adapter.AdapterInitListener
import com.particles.msp.adapter.AdapterInitStatus
import com.particles.msp.adapter.InitializationParameters
import com.particles.msp.api.MSPConstants
import com.particles.msp.api.MSPInitListener
import com.particles.msp.api.MSPInitStatus
import com.particles.msp.auction.AdConfigManager
import com.particles.msp.auction.BidderProvider
import com.particles.msp.util.Logger
import com.particles.msp.util.UserId
import com.particles.msp.util.UserIdRequest
import com.particles.msp.util.getAdvertisingId
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

object MSPManager {
    var bidLoaderProvider: BidLoaderProvider? = null
    var adNetworkAdapterProvider: AdNetworkAdapterProvider? = null
    var bidderProvider: BidderProvider? = null
    var org = ""
        private set
    var app = ""
        private set

    fun init(initParams: InitializationParameters,
             sdkInitListener: MSPInitListener,
             context: Context
    ) {
        org = initParams.getOrgId().toString()
        app = initParams.getAppId().toString()

        val adapters = AdNetwork.entries.mapNotNull {
            val adapter = adNetworkAdapterProvider?.getAdNetworkAdapter(it)
            if (adapter == null) {
                Logger.info("$it: adapter not found")
            } else {
                Logger.info("$it: adapter instance created successfully")
            }
            adapter
        }

        val adapterInitListeners = object : AdapterInitListener {
            private val results = mutableMapOf<AdNetwork, AdapterInitStatus>()

            override fun onComplete(
                adNetwork: AdNetwork,
                adapterInitStatus: AdapterInitStatus,
                message: String
            ) {
                when (adapterInitStatus) {
                    AdapterInitStatus.SUCCESS -> Logger.info("adapter $adNetwork is initialized: " + adapterInitStatus.message)
                    AdapterInitStatus.FAILURE -> Logger.info("adapter $adNetwork failed to init: $message")
                }

                results[adNetwork] = adapterInitStatus
                if (results.size == adapters.size) {
                    sdkInitListener.onComplete(MSPInitStatus.SUCCESS, MSPInitStatus.SUCCESS.message)
                }
            }
        }

        adapters.forEach {
            it.initialize(initParams, adapterInitListeners, context)
        }

        AdConfigManager.initialize(context, initParams)
        if (initParams.getOrgId() > 0) {
            // orgId: 0 is for in-house(Particle Media Inc.) App
            // for in-house app, we use the user ID provided by the app
            UserId.initialize(context)
            UserId.getCachedUserId()?.let {
                Logger.debug("Cached User ID: $it")
            } ?: run {
                if (initParams.getAppId() >= 0) {
                    Logger.debug("No cached User ID. Fetching from server...")
                    CoroutineScope(Dispatchers.Main).launch {
                        val request = UserIdRequest(
                            orgID = initParams.getOrgId(),
                            appID = initParams.getAppId(),
                            ppid = initParams.getParameters()[MSPConstants.INIT_PARAM_KEY_PPID] as? String
                                ?: "",
                            device_id = getAdvertisingId(context) ?: "",
                            email = initParams.getParameters()[MSPConstants.INIT_PARAM_KEY_EMAIL] as? String
                                ?: "",
                            token = initParams.getPrebidAPIKey()
                        )
                        val userId = UserId.fetchAndCacheUserId(request)
                        if (userId != null) {
                            Logger.debug("User ID fetched and cached: $userId")
                        } else {
                            Logger.debug("Failed to fetch user ID.")
                        }
                    }
                } else {
                    Logger.debug("No appId provided. Skipping User ID fetching.")
                }
            }
        } else {
            Logger.debug("Skipping User ID fetching for in-house App: orgId = ${initParams.getOrgId()}.")
        }
    }
}