package com.particles.msp.api

import com.particles.msp.AdCache
import com.particles.msp.MSPManager
import com.particles.msp.auction.AdConfigManager
import com.particles.msp.auction.Auction
import com.particles.msp.auction.AuctionBid
import com.particles.msp.auction.AuctionListener
import com.particles.msp.util.Logger
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

class AdLoader {
    fun loadAd(placementId: String, adListener: AdListener, adRequest: AdRequest) {
        val biddersInfo = AdConfigManager.getAdConfig(placementId)
        val bidderList = biddersInfo?.bidders?.mapNotNull { MSPManager.bidderProvider?.getBidder(it) } ?: emptyList()
        Logger.info("Starting auction for placementId: $placementId. requestId: ${adRequest.uuid}")
        Auction(bidderList, adRequest, false, biddersInfo?.auctionTimeout ?: 8000).startAuction(object : AuctionListener {
            override fun onSuccess(winningBid: AuctionBid) {
                CoroutineScope(Dispatchers.Main).launch {
                    adListener.onAdLoaded(adRequest.placementId)
                }
            }

            override fun onError(error: String) {
                CoroutineScope(Dispatchers.Main).launch {
                    adListener.onError("No winning bid from auction: $error")
                }
            }

        }, adListener)
    }

    fun getAd(placementId: String): MSPAd? {
        val winnerAuctionBid = runBlocking { getWinnerBidFromCache(placementId) }
        return winnerAuctionBid?.let {
            AdCache.getAd(it.bidderPlacementId)
        }
    }

    private suspend fun getWinnerBidFromCache(placementId: String): AuctionBid? = suspendCoroutine { continuation ->
        var isResumed = false
        fun safeResume(value: AuctionBid?) {
            if (!isResumed) {
                isResumed = true
                continuation.resume(value)
            } else {
                Logger.warning("[AdLoader: getAd] Attempted to resume continuation more than once. Ignoring.")
            }
        }

        val biddersInfo = AdConfigManager.getAdConfig(placementId)
        val bidderList = biddersInfo?.bidders?.mapNotNull { MSPManager.bidderProvider?.getBidder(it) } ?: emptyList()
        Auction(bidderList, null, true, 8000).startAuction(object : AuctionListener {
            override fun onSuccess(winningBid: AuctionBid) {
                safeResume(winningBid) // Resume coroutine with the result
            }

            override fun onError(error: String) {
                safeResume(null) // Resume coroutine with the result
            }

        }, null)
    }
}
