package com.particles.msp.auction

import android.content.Context
import com.particles.msp.adapter.InitializationParameters
import com.particles.msp.util.Logger
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import retrofit2.http.Body
import retrofit2.http.POST

data class AdConfigRequest(
    val app_id: Int,
    val org_id: Int,
    val token: String
)

data class AdConfigResponse(
    val message: String,
    val code: Int,
    val ad_config_settings: AdConfigSettings?
)

data class AdConfigSettings(
    val org_id: Int,
    val app_id: Int,
    val ad_config: String
)

interface AdConfigApi {
    @POST("getAdConfig") // Endpoint of the API
    suspend fun fetchAdConfig(@Body request: AdConfigRequest): AdConfigResponse
}


object AdConfigManager {
    private val retrofit: Retrofit by lazy {
        Retrofit.Builder()
            .baseUrl("https://msp-platform.newsbreak.com/")
            .addConverterFactory(GsonConverterFactory.create())
            .build()
    }
    private val api: AdConfigApi by lazy { retrofit.create(AdConfigApi::class.java) }

    private lateinit var appContext: Context
    private val jsonParser = Json { ignoreUnknownKeys = true }
    @Volatile private var cachedAdConfig: AdConfig? = null
    private const val configFileName = "msp_ad_config.json"

    fun initialize(context: Context, initParams: InitializationParameters) {
        this.appContext = context.applicationContext

        cachedAdConfig = loadConfigFromLocalStorage()
        Logger.info("ad config loaded from local file: $cachedAdConfig")

        CoroutineScope(Dispatchers.Main).launch {
            Logger.info("Fetching ad config from remote server....")
            fetchAdConfigFromServer(initParams.getAppId(), initParams.getOrgId(), initParams.getPrebidAPIKey())
        }
    }

    private suspend fun fetchAdConfigFromServer(
        appId: Int,
        orgId: Int,
        prebidApiKey: String
    ) = withContext(Dispatchers.IO) {
        try {
            val request = AdConfigRequest(
                app_id = appId,
                org_id = orgId,
                token = prebidApiKey
            )

            val response = api.fetchAdConfig(request)

            if (response.code == 0) {
                response.ad_config_settings?.let { settings ->
                    Logger.info("Ad config returned from remote server: ${settings.ad_config}")
                    val adConfig = jsonParser.decodeFromString<AdConfig>(settings.ad_config)
                    saveConfigToLocalStorage(adConfig)
                    cachedAdConfig = adConfig
                } ?: Logger.error("Ad Config settings are null.")
            } else {
                Logger.error("Failed to fetch ad config: ${response.message}")
            }
        } catch (e: Exception) {
            Logger.error("Exception occurred: ${e.message}")
        }
    }

    fun getAdConfig(placementId: String): Placement? {
        return cachedAdConfig?.placements?.find { it.placementId == placementId }
    }

    private fun saveConfigToLocalStorage(adConfig: AdConfig) {
        Logger.info("Save ad config returned from remote server to local file: $adConfig")
        val configJson = jsonParser.encodeToString(adConfig)
        appContext.openFileOutput(configFileName, Context.MODE_PRIVATE).use { it.write(configJson.toByteArray()) }
    }

    private fun loadConfigFromLocalStorage(): AdConfig? {
        return try {
            val configJson = appContext.openFileInput(configFileName).bufferedReader().use { it.readText() }
            jsonParser.decodeFromString(configJson)
        } catch (e: Exception) {
            null
        }
    }

    //private fun loadConfigFromRawResource(): AdConfig {
    //    val rawJson = loadJSONFromRaw(appContext, R.raw.local_ad_config)
    //    return jsonParser.decodeFromString(rawJson)
    //}
}

