package com.particles.msp.auction

import com.particles.msp.AdCache
import com.particles.msp.api.AdListener
import com.particles.msp.api.AdRequest
import com.particles.msp.util.Logger
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.async
import kotlinx.coroutines.awaitAll
import kotlinx.coroutines.launch
import kotlinx.coroutines.withTimeoutOrNull
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

class Auction(
    private val bidders: List<Bidder>,
    private val adRequest: AdRequest?,
    private val cacheOnly: Boolean,
    private val timeout: Int
) {
    fun startAuction(auctionListener: AuctionListener, adListener: AdListener?) {
        Logger.info("[Auction: ${if (cacheOnly) "Get Ad" else "Load Ad" }] started.")
        CoroutineScope(Dispatchers.IO).launch {
            val bidDeferreds = bidders.map { bidder ->
                async { fetchBid(bidder, adListener, cacheOnly) }
            }

            val winningBid = bidDeferreds.map { deferred ->
                async { withTimeoutOrNull(timeout.toLong()) { deferred.await() } }
            }.awaitAll()
                .filterNotNull()
                .maxByOrNull { it.ecpm }

            if (winningBid != null) {
                Logger.info("[Auction: ${if (cacheOnly) "Get Ad" else "Load Ad" }] completed. winner: (${winningBid.bidderName}, ${winningBid.ecpm}, ${winningBid.bidderPlacementId})")
                auctionListener.onSuccess(winningBid)
            } else {
                Logger.info("[Auction: ${if (cacheOnly) "Get Ad" else "Load Ad" }] completed. No winning bids")
                auctionListener.onError("No bids received")
            }
        }
    }

    private suspend fun fetchBid(bidder: Bidder, adListener: AdListener?, cacheOnly: Boolean): AuctionBid? = suspendCoroutine { continuation ->
        var isResumed = false
        fun safeResume(value: AuctionBid?) {
            if (!isResumed) {
                isResumed = true
                continuation.resume(value)
            } else {
                Logger.warning("[Auction: ${if (cacheOnly) "Get Ad" else "Load Ad" }] Attempted to resume continuation more than once. Ignoring.")
            }
        }

        if (!cacheOnly) {
            Logger.info("[Auction: Load Ad] Fetching bid from bidder: ${bidder.name}. bidderPlacementId: ${bidder.bidderPlacementId}")
        }
        val cachedAd = AdCache.peakAd(bidder.bidderPlacementId)
        if (cacheOnly || cachedAd != null) {
            cachedAd?.let {
                Logger.info("[Auction: ${if (cacheOnly) "Get Ad" else "Load Ad" }] Ads filled from cache: ${bidder.name}. price: ${it.adInfo["price"]}. bidderPlacementId: ${bidder.bidderPlacementId}")
                safeResume(AuctionBid(bidder.name, bidder.bidderPlacementId, it.adInfo["price"] as Double))
            } ?: run {
                Logger.info("[Auction: Get Ad ] Ads missed from cache: ${bidder.name}. bidderPlacementId: ${bidder.bidderPlacementId}")
                safeResume(null)
            }
        } else {
            adRequest?.let { adRequest ->
               adListener?.let { adListener ->
                   bidder.requestBid(adRequest, object : AuctionBidListener {
                       override fun onSuccess(bid: AuctionBid) {
                           Logger.info("[Auction] Ads filled from bidder: ${bidder.name}. bidderPlacementId: ${bidder.bidderPlacementId}")
                           safeResume(bid)
                       }

                       override fun onError(error: String) {
                           Logger.info("[Auction] Ads No fill from bidder: ${bidder.name}. bidderPlacementId: ${bidder.bidderPlacementId}")
                           safeResume(null)
                       }
                   }, adListener)
               }
            } ?: run {
                Logger.info("[Auction] adRequest or adListener is null. Bidder: ${bidder.name}. bidderPlacementId: ${bidder.bidderPlacementId}")
                safeResume(null)
            }
        }
    }
}