package com.particles.msp

import android.adservices.adid.AdId
import android.content.Context
import com.particles.msp.adapter.AdNetwork
import com.particles.msp.adapter.AdapterInitListener
import com.particles.msp.adapter.AdapterInitStatus
import com.particles.msp.adapter.InitializationParameters
import com.particles.msp.api.MSPConstants
import com.particles.msp.api.MSPInitListener
import com.particles.msp.api.MSPInitStatus
import com.particles.msp.util.Logger
import com.particles.msp.util.UserId
import com.particles.msp.util.UserIdRequest
import com.particles.msp.util.getAdvertisingId
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

object MSPManager {
    var bidLoaderProvider: BidLoaderProvider? = null
    var adNetworkAdapterProvider: AdNetworkAdapterProvider? = null

    fun init(initParams: InitializationParameters,
             sdkInitListener: MSPInitListener,
             context: Context
    ) {
        val nWaitingForCallbacks = adNetworkAdapterProvider?.getAdNetworkAdaptersCount()
        nWaitingForCallbacks?.let {
            val adNetworks = arrayOf(AdNetwork.Google, AdNetwork.Prebid, AdNetwork.Facebook, AdNetwork.Nova)
            var numWaitingForCallbacks = it
            val adapterInitListeners = object : AdapterInitListener {
                override fun onComplete(adNetwork: AdNetwork, adapterInitStatus: AdapterInitStatus, message: String) {
                    Logger.info("adapter $adNetwork is initialized: " + adapterInitStatus.message)
                    if (--numWaitingForCallbacks == 0) {
                        sdkInitListener.onComplete(MSPInitStatus.SUCCESS, MSPInitStatus.SUCCESS.message)
                    }
                }
            }
            for (adNetwork in adNetworks) {
                adNetworkAdapterProvider?.getAdNetworkAdapter(adNetwork)
                    ?.initialize(initParams, adapterInitListeners, context)
            }
        }

        UserId.initialize(context)
        UserId.getCachedUserId()?.let {
            Logger.debug("Cached User ID: $it")
        } ?: run {
            Logger.debug("No cached User ID. Fetching from server...")
            CoroutineScope(Dispatchers.Main).launch {
                val request = UserIdRequest(
                    orgID = initParams.getOrgId(),
                    appID = initParams.getAppId(),
                    ppid = initParams.getParameters()[MSPConstants.INIT_PARAM_KEY_PPID] as? String ?: "",
                    device_id = getAdvertisingId(context)?:"",
                    email = initParams.getParameters()[MSPConstants.INIT_PARAM_KEY_EMAIL] as? String ?: "",
                    token = initParams.getPrebidAPIKey()
                )
                val userId = UserId.fetchAndCacheUserId(request)
                if (userId != null) {
                    Logger.debug("User ID fetched and cached: $userId")
                } else {
                    Logger.debug("Failed to fetch user ID.")
                }
            }
        }
    }
}