package com.particles.msp

import ai.themsp.mspcore.BuildConfig
import android.Manifest
import android.content.Context
import android.content.Intent
import android.text.TextUtils
import androidx.annotation.RequiresPermission
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.ProcessLifecycleOwner
import com.particles.mes.android.MesTracker
import com.particles.mes.android.data.MesUserSignalEvent
import com.particles.mes.android.data.UserSignalType
import com.particles.msp.adapter.AdNetwork
import com.particles.msp.adapter.AdNetworkAdapter
import com.particles.msp.adapter.AdapterInitListener
import com.particles.msp.adapter.AdapterInitStatus
import com.particles.msp.adapter.InitializationParameters
import com.particles.msp.api.MSPConstants
import com.particles.msp.api.MSPInitListener
import com.particles.msp.api.MSPInitStatus
import com.particles.msp.auction.AdConfigManager
import com.particles.msp.auction.BidderProvider
import com.particles.msp.debug.DebugActivity
import com.particles.msp.util.AdIdRepo
import com.particles.msp.util.Logger
import com.particles.msp.util.MesTrackerExt.trackSdkInit
import com.particles.msp.util.UserId
import com.particles.msp.util.UserIdRequest
import com.particles.msp.util.getAdvertisingId
import com.particles.msp.util.getAppSignal
import com.particles.msp.util.getDeviceSignal
import com.particles.msp.util.getMesTracker
import com.particles.msp.util.getSDKSignal
import com.particles.msp.util.isAppFirstLaunch
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

object MSPManager: DefaultLifecycleObserver {
    var bidLoaderProvider: BidLoaderProvider? = null
    var adNetworkAdapterProvider: AdNetworkAdapterProvider? = null
    var bidderProvider: BidderProvider? = null
    var org = ""
        private set
    var app = ""
        private set
    var ppid = ""
        private set
    var adapterMetadataMap: Map<AdNetwork, AdNetworkAdapter.Metadata>? = null
        private set

    var mesTracker: MesTracker? = null
        private set

    var mesMspUserId: String = ""
        private set
        get() {
            return if (TextUtils.isEmpty(field)) {
                UserId.getCachedUserId()
            } else {
                field
            }
        }

    private lateinit var appContext: Context

    private var initLatencyMs = 0L

    const val version: String = BuildConfig.VERSION_NAME

    fun init(initParams: InitializationParameters,
             sdkInitListener: MSPInitListener,
             context: Context,
             initStartTimeMs: Long
    ) {
        appContext = context.applicationContext
        org = initParams.getOrgId().toString()
        app = initParams.getAppId().toString()
        ppid = initParams.getParameters()[MSPConstants.INIT_PARAM_KEY_PPID] as? String ?: ""
        mesTracker = getMesTracker(initParams.getMesHostUrl())
        mesMspUserId = initParams.getParameters()[MSPConstants.INIT_PARAM_KEY_PPID] as? String ?: ""

        val metadataMap = mutableMapOf<AdNetwork, AdNetworkAdapter.Metadata>()
        val adapters = AdNetwork.entries.mapNotNull {
            val adapter = adNetworkAdapterProvider?.getAdNetworkAdapter(it)
            if (adapter == null) {
                Logger.info("$it: adapter not found")
            } else {
                Logger.info("$it: adapter instance created successfully")
                adapter.metadata?.let { metadata -> metadataMap[it] = metadata }
            }
            adapter
        }

        adapterMetadataMap = metadataMap
        val adaptersInitCompleteTimeMs = mutableMapOf<AdNetwork, Long>()
        val adaptersInitStartTimeMs = mutableMapOf<AdNetwork, Long>()
        val adapterInitListeners = object : AdapterInitListener {
            private val results = mutableMapOf<AdNetwork, AdapterInitStatus>()

            @RequiresPermission(Manifest.permission.ACCESS_NETWORK_STATE)
            override fun onComplete(
                adNetwork: AdNetwork,
                adapterInitStatus: AdapterInitStatus,
                message: String
            ) {
                when (adapterInitStatus) {
                    AdapterInitStatus.SUCCESS -> Logger.info("adapter $adNetwork is initialized: " + adapterInitStatus.message)
                    AdapterInitStatus.FAILURE -> Logger.info("adapter $adNetwork failed to init: $message")
                }

                results[adNetwork] = adapterInitStatus
                adaptersInitStartTimeMs[adNetwork]?.let {
                    adaptersInitCompleteTimeMs[adNetwork] = System.currentTimeMillis() - it
                    Logger.info("Initializing adapter ${adNetwork.name} complete. duration:${adaptersInitCompleteTimeMs[adNetwork]} ms")
                }

                if (results.size == adapters.size) {
                    handleAllAdaptersComplete(sdkInitListener, initStartTimeMs, adaptersInitCompleteTimeMs, context,
                        UserId.getCachedUserId(), ppid)
                }
            }
        }

        adapters.forEach {
            Logger.info("Initializing adapter start: ${it::class.simpleName}")
            adaptersInitStartTimeMs[it.adNetwork] = System.currentTimeMillis()
            it.initialize(initParams, adapterInitListeners, context)
        }

        AdConfigManager.initialize(context, initParams)
        UserId.initialize(context)
        if (!TextUtils.isEmpty(UserId.getCachedUserId())) {
            Logger.debug("Cached User ID: ${UserId.getCachedUserId()}")
        } else {
            if (initParams.getAppId() >= 0) {
                Logger.debug("No cached User ID. Fetching from server...")
                CoroutineScope(Dispatchers.Main).launch {
                    val request = UserIdRequest(
                        orgID = initParams.getOrgId(),
                        appID = initParams.getAppId(),
                        ppid = initParams.getParameters()[MSPConstants.INIT_PARAM_KEY_PPID] as? String
                            ?: "",
                        device_id = getAdvertisingId(context) ?: "",
                        email = initParams.getParameters()[MSPConstants.INIT_PARAM_KEY_EMAIL] as? String
                            ?: "",
                        token = initParams.getPrebidAPIKey()
                    )
                    val userId = UserId.fetchAndCacheUserId(request)
                    if (!TextUtils.isEmpty(userId)) {
                        Logger.debug("User ID fetched and cached: $userId")
                    } else {
                        Logger.debug("Failed to fetch user ID.")
                    }
                }
            } else {
                Logger.debug("No appId provided. Skipping User ID fetching.")
            }
        }

        ProcessLifecycleOwner.get().lifecycle.addObserver(this)
        initLatencyMs = System.currentTimeMillis() - initStartTimeMs
        Logger.info("MSP init latency: $initLatencyMs ms")
    }

    @RequiresPermission(Manifest.permission.ACCESS_NETWORK_STATE)
    private fun handleAllAdaptersComplete(
        listener: MSPInitListener,
        initStartTsMs: Long,
        completeTimeMs: Map<AdNetwork, Long>,
        context: Context,
        mspId: String,
        ppid: String
    ) {
        listener.onComplete(MSPInitStatus.SUCCESS, MSPInitStatus.SUCCESS.message)

        val totalCompleteTime = System.currentTimeMillis() - initStartTsMs

        Logger.info("Initializing MSP complete. duration:$totalCompleteTime ms")

        val completeTimeByAdNetwork =
            completeTimeMs.map { (adNetwork, completeTimeLong) ->
                adNetwork.name to completeTimeLong.toInt()
            }.toMap()



        CoroutineScope(Dispatchers.Main).launch {
            val adId = AdIdRepo.get(context)
            mesTracker?.trackSdkInit(
                org,
                app,
                initLatencyMs.toInt(),
                totalCompleteTime.toInt(),
                completeTimeByAdNetwork,
                mspId,
                context,
                adId,
                ppid,
                getSDKSignal(context),
                getDeviceSignal(context),
                getAppSignal()
            )

            if (isAppFirstLaunch(context)) {
                mesTracker?.trackUserSignal(MesUserSignalEvent(
                    type = UserSignalType.USER_SIGNAL_TYPE_INTO_ATTRIBUTION,
                    sdkSignal = getSDKSignal(context),
                    deviceSignal = getDeviceSignal(context = appContext),
                    appSignal = getAppSignal()
                ))
            }
        }
    }

    fun showMediationDebugger(
        context: Context,
    ) {
        val intent = Intent(context, DebugActivity::class.java)
        context.startActivity(intent)
    }

    fun setPpid(ppid: String) {
        this.ppid = ppid
        this.mesMspUserId = ppid
    }

    override fun onStart(owner: LifecycleOwner) {
        Logger.info("app moved to foreground")
        CoroutineScope(Dispatchers.Main).launch {
            mesTracker?.trackUserSignal(MesUserSignalEvent(
                type = UserSignalType.USER_SIGNAL_TYPE_INTO_FOREGROUND,
                sdkSignal = getSDKSignal(context = appContext),
                deviceSignal = getDeviceSignal(context = appContext),
                appSignal = getAppSignal()
            ))
        }
    }

    override fun onStop(owner: LifecycleOwner) {
        Logger.info("app moved to background")
        CoroutineScope(Dispatchers.Main).launch {
            mesTracker?.trackUserSignal(MesUserSignalEvent(
                type = UserSignalType.USER_SIGNAL_TYPE_INTO_BACKGROUND,
                sdkSignal = getSDKSignal(context = appContext),
                deviceSignal = getDeviceSignal(context = appContext),
                appSignal = getAppSignal()
            ))
        }
    }
}
