package com.particles.msp.adapter

import android.content.Context
import com.particles.mes.protos.ErrorCode
import com.particles.msp.AdCache
import com.particles.msp.MSPManager
import com.particles.msp.api.AdListener
import com.particles.msp.api.AdRequest
import com.particles.msp.api.MSPAd
import com.particles.msp.api.NativeAd
import com.particles.msp.auction.AuctionBid
import com.particles.msp.auction.AuctionBidListener
import com.particles.msp.auction.BidderInfo
import com.particles.msp.util.AD_INFO_NETWORK_AD_UNIT_ID
import com.particles.msp.util.AD_INFO_NETWORK_CREATIVE_ID
import com.particles.msp.util.AD_INFO_NETWORK_NAME
import com.particles.msp.util.AD_INFO_OPENRTB_BURL
import com.particles.msp.util.AD_INFO_OPENRTB_NURL
import com.particles.msp.util.Logger
import com.particles.msp.util.MesTrackerExt.trackAdClick
import com.particles.msp.util.MesTrackerExt.trackAdHide
import com.particles.msp.util.MesTrackerExt.trackAdImpression
import com.particles.msp.util.MesTrackerExt.trackAdReport
import com.particles.msp.util.MesTrackerExt.trackGetAd
import com.particles.msp.util.MesTrackerExt.trackLoadAd
import com.particles.msp.util.getBuckets
import com.particles.msp.util.getInferredCountry
import org.prebid.mobile.rendering.bidding.data.bid.BidResponse

abstract class AdNetworkAdapter {
    protected var mspAd: MSPAd? = null
    private var adRequest: AdRequest? = null
    protected var bidResponse: BidResponse? = null
    private var bidderInfo: BidderInfo? = null

    abstract val adNetworkName: String
    abstract val adNetwork: AdNetwork

    data class Metadata(
        val sdkVersion: String,
    )

    open val metadata: Metadata? = null

    open fun loadAdCreative(bidResponse: BidResponse?, auctionBidListener: AuctionBidListener, context: Context, adRequest: AdRequest, adListener: AdListener,
                            bidderPlacementId: String,
                            bidderInfo: BidderInfo) {
        this.adRequest = adRequest
        this.bidResponse = bidResponse
        this.bidderInfo = bidderInfo
    }

    open fun initialize(
        initParams: InitializationParameters,
        adapterInitListener: AdapterInitListener,
        context: Context) {
    }

    open fun isAdInvalid(): Boolean = false

    abstract fun destroyAd()
    abstract fun prepareViewForInteraction(nativeAd: NativeAd, nativeAdView: Any)

    fun sendHideAdEvent(reason: String, areaScreenshot: ByteArray?, fullScreenshot: ByteArray?) {
        mspAd?.let {
            MSPManager.mesTracker?.trackAdHide(
                adRequest,
                bidResponse,
                it,
                reason,
                areaScreenshot,
                fullScreenshot
            )
        }
    }

    fun sendReportAdEvent(reason: String, description: String?, areaScreenshot: ByteArray?, fullScreenshot: ByteArray?) {
        mspAd?.let {
            MSPManager.mesTracker?.trackAdReport(
                adRequest,
                bidResponse,
                it,
                reason,
                description,
                areaScreenshot,
                fullScreenshot
            )
        }
    }

    fun trackGetAdSuccessEvent() {
        mspAd?.let {
            MSPManager.mesTracker?.trackGetAd(
                adRequest,
                bidResponse,
                it,
                ErrorCode.ERROR_CODE_SUCCESS
            )
        }
    }

    fun trackLoadAdSuccessEvent(
        latency: Int,
        loadFromCache: Boolean,
    ) {
        adRequest?.let {
            MSPManager.mesTracker?.trackLoadAd(
                it,
                bidResponse,
                mspAd,
                latency,
                ErrorCode.ERROR_CODE_SUCCESS,
                loadFromCache
            )
        }
    }

    fun handleAdLoadError(
        auctionBidListener: AuctionBidListener,
        bidderPlacementId: String,
        error: String
    ) {
        auctionBidListener.onError(error)
    }

    private fun replaceMacroAuctionPrice(url: String?, price: Double): String? {
        return url?.replace("\${AUCTION_PRICE}", price.toString())
    }

    fun handleAdLoaded(
        ad: MSPAd,
        auctionBidListener: AuctionBidListener,
        bidderPlacementId: String,
        adNetworkPlacementId: String,
        price: Double
    ) {
        mspAd = ad
        mspAd?.apply {
            adInfo[AD_INFO_NETWORK_NAME] = adNetworkName
            adInfo[AD_INFO_NETWORK_AD_UNIT_ID] = adNetworkPlacementId
            adInfo[AD_INFO_NETWORK_CREATIVE_ID] = bidResponse?.winningBid?.crid ?: "unknown"
            replaceMacroAuctionPrice(bidResponse?.winningBid?.nurl, price)?.let {
                adInfo[AD_INFO_OPENRTB_NURL] = it
            }
            replaceMacroAuctionPrice(bidResponse?.winningBid?.burl, price)?.let {
                adInfo[AD_INFO_OPENRTB_BURL] = it
            }
            adInfo["client_bidder"] = bidderInfo?.name?: ""
            adInfo["adnetwork_placement_id"] = adNetworkPlacementId
            adInfo["ad_loaded_at"] = System.currentTimeMillis()
            adInfo["price"] = price
        }

        Logger.verbose("[Adapter: Base] Save Ad to cache: bidderPlacementId: $bidderPlacementId")
        AdCache.saveAd(bidderPlacementId, ad)
        auctionBidListener.onSuccess(AuctionBid("msp", bidderPlacementId, ad.adInfo["price"] as Double, false))
    }

    fun handleAdClicked(listener: AdListener, adRequest: AdRequest) {
        mspAd?.let {
            listener.onAdClicked(it)
            MSPManager.mesTracker?.trackAdClick(
                adRequest,
                bidResponse,
                it
            )
        }
    }

    fun handleAdDismissed(listener: AdListener, adRequest: AdRequest) {
        mspAd?.let {
            listener.onAdDismissed(it)
        }
    }

    fun handleAdImpression(listener: AdListener, adRequest: AdRequest) {
        mspAd?.let {
            listener.onAdImpression(it)
            MSPManager.mesTracker?.trackAdImpression(
                adRequest,
                bidResponse,
                it
            )
        }
    }
}