package com.particles.msp.debug

import ai.themsp.mspcore.R
import android.os.Bundle
import android.view.Gravity
import android.view.View
import android.widget.FrameLayout
import android.widget.TextView
import androidx.appcompat.app.AppCompatActivity
import com.particles.msp.api.AdFormat
import com.particles.msp.api.AdLoader
import com.particles.msp.api.AdRequest
import com.particles.msp.api.AdSize
import com.particles.msp.api.BannerAdView
import com.particles.msp.api.MSPAd
import com.particles.msp.api.MSPConstants
import com.particles.msp.api.NativeAd
import com.particles.msp.api.NativeAdView
import com.particles.msp.api.NativeAdViewBinder
import com.particles.msp.util.Logger

internal class AdActivity : AppCompatActivity() {
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_ad)
        loadAd()
    }

    private fun loadAd() {
        val network = intent.getStringExtra(KEY_NETWORK) ?: NOVA
//        val format = intent.getStringExtra(KEY_FORMAT) ?: INTERSTITIAL
        val creativeType = intent.getStringExtra(KEY_CREATIVE_TYPE) ?: VIDEO
        val layout = intent.getStringExtra(KEY_LAYOUT) ?: VERTICAL
        val highEngagement = intent.getStringExtra(KEY_HIGH_ENGAGEMENT) ?: NO
        val placementId = intent.getStringExtra(KEY_PLACEMENT_ID) ?: ""

        val adViewContainer = findViewById<FrameLayout>(R.id.ad_container)
        adViewContainer.removeAllViews()
        adViewContainer.addView(TextView(this).apply {
            text = resources.getText(R.string.ad_loading_text)
            textSize = 20f
            layoutParams = FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.WRAP_CONTENT,
                FrameLayout.LayoutParams.WRAP_CONTENT
            ).apply {
                gravity = Gravity.CENTER
            }
        })

        // 2. listen and handle loaded Ad
        val adLoadListener = object : AdListenerAdapter() {

            override fun onAdLoaded(placementId: String) {
                adViewContainer.removeAllViews()
                Logger.info(TAG, "Ad load event received. placementId: $placementId. Fetching ads from cache...")
                val ad: MSPAd? = AdLoader().getAd(placementId)
                if (ad == null) {
                    Logger.info(TAG, "Got null ad from cache. placementId: $placementId")
                    return
                }

                Logger.info(TAG, "Got ad from cache. placementId: $placementId. adInfo: ${ad.adInfo}")

                var adView: View? = null
                when (ad) {
                    is BannerAdView -> {
                        adView = (ad.adView) as View
                        Logger.info("Banner Ad loaded.")
                    }
                    is NativeAd -> {
                        Logger.info(TAG, "Native Ad loaded. Title: ${ad.title}")
                        val nativeAdViewBinder: NativeAdViewBinder = NativeAdViewBinder.Builder()
                            .layoutResourceId(R.layout.native_ad)
                            .titleTextViewId(R.id.ad_title_text_view)
                            .advertiserTextViewId(R.id.advertiser_text_view)
                            .bodyTextViewId(R.id.ad_body_text_view)
                            .callToActionViewId(R.id.cta_button)
                            .mediaViewId(R.id.ad_media_view_container)
                            .optionsViewId(R.id.options_view)
                            .iconViewId(R.id.icon_image_view)
                            .build()
                        adView = NativeAdView(ad, nativeAdViewBinder, applicationContext)
                    }
                    else -> {
                        Logger.info(TAG, "not supported ad type: ${ad::class.java.simpleName}")
                    }
                }
                adView?.let {
                    adViewContainer.addView(it)
                    if (ad !is NativeAd || ad.iconUrl == null) {
                        adViewContainer.findViewById<View>(R.id.icon_image_view)?.visibility = View.GONE
                    }
                }
            }
        }

        val adRequest = AdRequest.Builder(AdFormat.MULTI_FORMAT)
            .setContext(applicationContext)
            .setPlacement(placementId)
            .setAdSize(AdSize(300, 250, isInlineAdaptiveBanner = false, isAnchorAdaptiveBanner = false))
            .setCustomParams(mapOf(
                MSPConstants.CUSTOM_PARAM_KEY_USER_ID to "177905312",
                //MSPConstants.AD_REQUEST_CUSTOM_PARAM_KEY_GOOGLE_MULTI_FORMAT to false
                //MSPConstants.GOOGLE_AD_CONTENT_URL to "https://h5.newsbreakapp.com/mp/0zsgziYv?cv=25.14.0&platform=1"
            ))
            .setAdaptiveBannerSize(AdSize(384, 0, isInlineAdaptiveBanner = true, isAnchorAdaptiveBanner = false))
            .setTestParams(getTestParams(network, creativeType, layout, highEngagement))
            .build()

        val start = System.currentTimeMillis()
        Logger.info(TAG, "AdLoader.loadAd() start.")
        AdLoader().loadAd(placementId, adLoadListener, adRequest)
        Logger.info(TAG, "AdLoader.loadAd() end. DURATION: ${System.currentTimeMillis() - start} ms")
    }
}