package com.particles.msp.util

import android.content.Context
import android.content.SharedPreferences
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import retrofit2.http.Body
import retrofit2.http.Headers
import retrofit2.http.POST

data class UserIdResponse(
    val id_str: String,
    val code: Int,
    val message: String
)

data class UserIdRequest(
    val orgID: Int,
    val appID: Int,
    val ppid: String,
    val device_id: String,
    val email: String,
    val token: String
)

interface UserIdApi {
    @Headers("Content-Type: application/json")
    @POST("/getId")
    suspend fun fetchUserId(@Body request: UserIdRequest): UserIdResponse
}

object UserId {

    private const val USER_ID_CACHE_KEY = "cached_user_id"
    private const val USER_ID_STR_CACHE_KEY = "cached_user_id_str"
    private var preferences: SharedPreferences? = null
    private val retrofit: Retrofit by lazy {
        Retrofit.Builder()
            .baseUrl("https://id-msp.newsbreak.com")
            .addConverterFactory(GsonConverterFactory.create())
            .build()
    }
    private val api: UserIdApi by lazy { retrofit.create(UserIdApi::class.java) }

    fun initialize(context: Context) {
        preferences = context.getSharedPreferences("msp_sdk_userId", Context.MODE_PRIVATE)
    }

    suspend fun fetchAndCacheUserId(request: UserIdRequest): String = withContext(Dispatchers.IO) {
        try {
            val response = api.fetchUserId(request)
            return@withContext if (response.code == 0) {
                cacheUserId(response.id_str)
                response.id_str
            } else {
                Logger.error("Failed to fetch user ID: ${response.code} - ${response.message}")
                ""
            }
        } catch (e: Exception) {
            Logger.error("Failed to fetch user ID: exception - ${e.message}")
            ""
        }
    }

    fun getCachedUserId(): String {
        return preferences?.getLong(USER_ID_CACHE_KEY, -1L).takeIf { it != -1L }?.toString() ?:
        preferences?.getString(USER_ID_STR_CACHE_KEY, "") ?: ""
    }

    private fun cacheUserId(id: String) {
        preferences?.edit()?.putString(USER_ID_STR_CACHE_KEY, id)?.apply()
    }
}