package com.particles.msp.debug

import ai.themsp.mspcore.R
import android.content.Intent

import android.os.Bundle
import android.text.TextUtils
import android.view.View
import android.widget.AdapterView
import android.widget.ArrayAdapter
import android.widget.TextView
import android.widget.Toast
import androidx.activity.viewModels
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.widget.ListPopupWindow
import androidx.lifecycle.lifecycleScope
import com.google.android.material.snackbar.Snackbar
import com.particles.msp.api.AdFormat
import com.particles.msp.api.AdLoader
import com.particles.msp.api.AdRequest
import com.particles.msp.api.InterstitialAd
import com.particles.msp.api.MSPAd
import com.particles.msp.debug.datasource.AdDebugHardCodedDataSource
import com.particles.msp.debug.repository.AdDebugDataRepository
import com.particles.msp.debug.state.AdDebugInfoUiState
import com.particles.msp.debug.ui.TitledListView
import com.particles.msp.debug.viewmodel.AdDebugViewModel
import com.particles.msp.debug.viewmodel.AdDebugViewModelFactory
import com.particles.msp.util.Logger
import kotlinx.coroutines.launch
import kotlin.time.measureTime

internal class DebugActivity : AppCompatActivity() {

    private val repository = AdDebugDataRepository(AdDebugHardCodedDataSource())
    private val viewModelFactory: AdDebugViewModelFactory by lazy {
        AdDebugViewModelFactory(
            repository
        )
    }
    private val viewModel: AdDebugViewModel by viewModels { viewModelFactory }

    private var network: String = NOVA
    private var format: String = INTERSTITIAL
    private var creativeType: String = VIDEO
    private var layout: String = VERTICAL
    private var highEngagement: String = NO

    private var isNova: Boolean = false
    private var isInterstitial: Boolean = false
    private var placementId = ""

    private lateinit var tvPlacementId: TextView
    private lateinit var titledAdNetworkListView: TitledListView
    private lateinit var titledAdFormatListView: TitledListView
    private lateinit var titledAdCreativeTypeListView: TitledListView
    private lateinit var titledAdLayoutListView: TitledListView
    private lateinit var titledAdHighEngagementListView: TitledListView

    private lateinit var loadButton: View
    private lateinit var destroyButton: View

    private var ad: MSPAd? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_debugger)

        tvPlacementId = findViewById(R.id.tv_placement_id)
        titledAdNetworkListView = findViewById(R.id.ad_network)
        titledAdFormatListView = findViewById(R.id.ad_format)
        titledAdCreativeTypeListView = findViewById(R.id.ad_creative_type)
        titledAdLayoutListView = findViewById(R.id.ad_layout)
        titledAdHighEngagementListView = findViewById(R.id.ad_high_engagement)

        loadButton = findViewById(R.id.btn_load_ad)
        loadButton.setOnClickListener { loadAd() }
        destroyButton = findViewById(R.id.btn_destroy)
        destroyButton.setOnClickListener {
            ad?.destroy()
            ad = null
        }

        viewModel.fetchResources()

        observeUiState()
    }

    private fun getListPopupMenu(
        anchorView: View,
        itemList: List<String>,
    ): ListPopupWindow {
        return ListPopupWindow(
            applicationContext,
            null,
            androidx.appcompat.R.attr.listPopupWindowStyle
        ).apply {
            val adapter: ArrayAdapter<CharSequence> =
                ArrayAdapter<CharSequence>(
                    applicationContext,
                    R.layout.debugger_popup_item,
                    itemList
                )
            setAdapter(adapter)
            this.anchorView = anchorView
            setModal(true)
            setOnItemClickListener { parent: AdapterView<*>?, view: View?, position: Int, id: Long ->
                placementId = itemList[position]
                tvPlacementId.text = placementId
                dismiss()
            }
        }
    }

    private fun observeUiState() = lifecycleScope.launch {
        lifecycleScope.launchWhenStarted {
            viewModel.uiState.collect { state ->
                when (state) {
                    is AdDebugInfoUiState.AdDebugInfo -> handleAdDebugInfo(state)

                    is AdDebugInfoUiState.Error -> {
                        Logger.info(
                            TAG,
                            "an error occurred when collect uiState. " +
                                    "errorCode = ${state.code}, errorMsg = ${state.msg}"
                        )
                    }

                    AdDebugInfoUiState.Loading -> {}
                }
            }
        }
    }

    private fun handleAdDebugInfo(state: AdDebugInfoUiState.AdDebugInfo) {
        titledAdNetworkListView.setList(state.networks)
        titledAdFormatListView.setList(state.formats)
        titledAdCreativeTypeListView.setList(state.creativeTypes)
        titledAdLayoutListView.setList(state.layouts)
        titledAdHighEngagementListView.setList(state.highEngagements)

        val popupWindow = getListPopupMenu(tvPlacementId, state.placementIds)
        tvPlacementId.setOnClickListener {
            popupWindow.show()
        }

        isNova = state.networks.any { network ->
            network.selected && TextUtils.equals(
                network.text,
                NOVA
            )
        }
        isInterstitial = state.formats.any { format ->
            format.selected && TextUtils.equals(
                format.text,
                INTERSTITIAL
            )
        }

        network = state.networks.first { network -> network.selected }.value
        format = state.formats.first { format -> format.selected }.value
        creativeType = state.creativeTypes.first { creativeType -> creativeType.selected }.value
        layout = state.layouts.first { layout -> layout.selected }.value
        highEngagement =
            state.highEngagements.first { highEngagement -> highEngagement.selected }.value

        titledAdCreativeTypeListView.visibility = visibleOrGone(isNova)
        titledAdLayoutListView.visibility = visibleOrGone(isNova && isInterstitial)
        titledAdHighEngagementListView.visibility = visibleOrGone(isNova && isInterstitial)
    }

    private fun loadAd() {
        Logger.info(
            TAG,
            "begin loadAd: " +
                    "placementId = $placementId, " +
                    "network = $network, " +
                    "format = $format, " +
                    "creativeType = $creativeType, " +
                    "layout = $layout, " +
                    "highEngagement = $highEngagement"
        )

        if (placementId.isEmpty()) {
            Snackbar.make(
                this.findViewById<View>(android.R.id.content),
                resources.getText(R.string.debugger_lack_of_placement_hint),
                Snackbar.LENGTH_LONG
            )
                .show()
            return
        }

        when (format) {
            convertToValue(INTERSTITIAL) -> loadInterstitialAd()
            convertToValue(NATIVE) -> loadNativeAd()
            else -> {}
        }
    }

    private fun loadInterstitialAd() {
        val adLoadListener = object : AdListenerAdapter() {
            override fun onAdLoaded(placementId: String) {
                Logger.info(
                    TAG,
                    "Ad load event received. placementId: " +
                            "$placementId. Fetching ads from cache..."
                )
                val ad: MSPAd? = AdLoader().getAd(placementId)
                if (ad == null) {
                    Logger.info(TAG, "Got null ad from cache. placementId: $placementId")
                    return
                }

                Logger.info(TAG, "Got ad from cache. placementId: $placementId. adInfo: ${ad.adInfo}")

                Logger.info(TAG, "Interstitial Ad loaded.")

                (ad as? InterstitialAd)?.show(this@DebugActivity)
            }
        }

        val adRequest = AdRequest.Builder(AdFormat.INTERSTITIAL)
            .setContext(applicationContext)
            .setPlacement(placementId)
            .setCustomParams(
                mapOf(
                    "user_id" to "177905312",
                    //MSPConstants.AD_REQUEST_CUSTOM_PARAM_KEY_GOOGLE_MULTI_FORMAT to false,
                    //MSPConstants.GOOGLE_AD_CONTENT_URL to "https://h5.newsbreakapp.com/mp/0zsgziYv?cv=25.14.0&platform=1",
                    //REQUEST_PARAM_KEY_NOVA_ONLY to true
                )
            )
            .setTestParams(getTestParams(network, creativeType, layout, highEngagement))
            .build()

        Logger.info(TAG, "AdLoader.loadAd() start.")
        Toast.makeText(this, resources.getText(R.string.ad_loading_text), Toast.LENGTH_SHORT).show()
        val duration = measureTime { AdLoader().loadAd(placementId, adLoadListener, adRequest) }
        Logger.info(TAG, "AdLoader.loadAd() end. DURATION: $duration ms")
    }

    private fun loadNativeAd() {
        val intent = Intent(this, AdActivity::class.java).apply {
            putExtra(KEY_NETWORK, network)
            putExtra(KEY_FORMAT, format)
            putExtra(KEY_CREATIVE_TYPE, creativeType)
            putExtra(KEY_LAYOUT, layout)
            putExtra(KEY_HIGH_ENGAGEMENT, highEngagement)
            putExtra(KEY_PLACEMENT_ID, placementId)
        }
        startActivity(intent)
    }
}