package com.particles.msp.debug.viewmodel

import android.text.TextUtils
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.particles.msp.debug.repository.AdDebugDataRepository
import com.particles.msp.debug.AdDebugCategory
import com.particles.msp.debug.models.AdDebugResource
import com.particles.msp.debug.state.AdDebugInfoUiState
import com.particles.msp.debug.state.AdDebugItemUiState
import com.particles.msp.util.Logger
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import okio.IOException

internal class AdDebugViewModel(
    private val repository: AdDebugDataRepository
) : ViewModel() {

    private val _uiState: MutableStateFlow<AdDebugInfoUiState> =
        MutableStateFlow(AdDebugInfoUiState.AdDebugInfo())
    val uiState: StateFlow<AdDebugInfoUiState> = _uiState

    private var fetchJob: Job? = null

    fun fetchResources() {
        fetchJob?.cancel()
        fetchJob = viewModelScope.launch {
            try {
                val networks = getDebugItemUiState(AdDebugCategory.NETWORK)
                val formats = getDebugItemUiState(AdDebugCategory.FORMAT)
                val creativeTypes = getDebugItemUiState(AdDebugCategory.CREATIVE_TYPE)
                val layouts = getDebugItemUiState(AdDebugCategory.LAYOUT)
                val highEngagements = getDebugItemUiState(AdDebugCategory.HIGH_ENGAGEMENT)
                val placementIds = repository.fetchPlacementIds()
                _uiState.update {
                    AdDebugInfoUiState.AdDebugInfo(
                        placementIds = placementIds,
                        networks = networks,
                        formats = formats,
                        creativeTypes = creativeTypes,
                        layouts = layouts,
                        highEngagements = highEngagements,
                    )
                }
            } catch (ioe: IOException) {
                val message = ioe.message ?: "io exception"
                Logger.info("[Debugger] io exception thrown in fetchResources, error message: $message")
            }
        }
    }

    private suspend fun getDebugItemUiState(
        category: AdDebugCategory
    ): List<AdDebugItemUiState> {
        return repository
            .fetchAdResources(category)
            .toDebugItemUiState(category)
    }

    private fun List<AdDebugResource>.toDebugItemUiState(
        category: AdDebugCategory
    ): List<AdDebugItemUiState> {
        return this.map { resource ->
            AdDebugItemUiState(
                text = resource.text,
                value = resource.value,
                selected = resource.selected,
                onSelect = {
                    updateDebugInfo(category, it)
                }
            )
        }
    }

    fun updateDebugInfo(
        category: AdDebugCategory,
        text: String,
    ) {
        when (val state = _uiState.value) {
            is AdDebugInfoUiState.AdDebugInfo -> {
                when (category) {
                    AdDebugCategory.NETWORK -> updateNetwork(state, text)
                    AdDebugCategory.FORMAT -> updateFormat(state, text)
                    AdDebugCategory.CREATIVE_TYPE -> updateCreativeType(state, text)
                    AdDebugCategory.LAYOUT -> updateLayout(state, text)
                    AdDebugCategory.HIGH_ENGAGEMENT -> updateHighEngagement(state, text)
                }
            }

            else -> {
                Logger.info("[debugger] wrong state when updateDebugInfo. state = $state")
            }
        }
    }

    private fun updateNetwork(
        state: AdDebugInfoUiState.AdDebugInfo,
        text: String,
    ) {
        val networks = state.networks.map { network ->
            network.copy(
                selected = TextUtils.equals(network.text, text)
            )
        }
        _uiState.update {
            state.copy(networks = networks)
        }
    }

    private fun updateFormat(
        state: AdDebugInfoUiState.AdDebugInfo,
        text: String,
    ) {
        val formats = state.formats.map { format ->
            format.copy(
                selected = TextUtils.equals(format.text, text)
            )
        }
        _uiState.update {
            state.copy(formats = formats)
        }
    }

    private fun updateCreativeType(
        state: AdDebugInfoUiState.AdDebugInfo,
        text: String,
    ) {
        val creativeTypes = state.creativeTypes.map { creativeType ->
            creativeType.copy(
                selected = TextUtils.equals(creativeType.text, text)
            )
        }
        _uiState.update {
            state.copy(creativeTypes = creativeTypes)
        }
    }

    private fun updateLayout(
        state: AdDebugInfoUiState.AdDebugInfo,
        text: String,
    ) {
        val layouts = state.layouts.map { layout ->
            layout.copy(
                selected = TextUtils.equals(layout.text, text)
            )
        }
        _uiState.update {
            state.copy(layouts = layouts)
        }
    }

    private fun updateHighEngagement(
        state: AdDebugInfoUiState.AdDebugInfo,
        text: String,
    ) {
        val highEngagements = state.highEngagements.map { highEngagement ->
            highEngagement.copy(
                selected = TextUtils.equals(highEngagement.text, text)
            )
        }
        _uiState.update {
            state.copy(highEngagements = highEngagements)
        }
    }
}