package com.particles.msp.auction

import android.content.Context
import com.particles.msp.adapter.InitializationParameters
import com.particles.msp.api.AdRequest
import com.particles.msp.util.Logger
import com.particles.msp.util.UserId
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.serialization.InternalSerializationApi
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import retrofit2.http.Body
import retrofit2.http.POST
import kotlin.random.Random

data class AdConfigRequest(
    val app_id: Int,
    val org_id: Int,
    val token: String
)

data class AdConfigResponse(
    val message: String,
    val code: Int,
    val ad_config_settings: AdConfigSettings?,
    val log_config: LogConfig?
)

data class AdConfigSettings(
    val org_id: Int,
    val app_id: Int,
    val ad_config: String
)

data class LogConfig(
    val org_id: Int,
    val app_id: Int,
    val log_sample_rate: Double,
    val msp_id_whitelist: List<String>
)

interface AdConfigApi {
    @POST("getAdConfig") // Endpoint of the API
    suspend fun fetchAdConfig(@Body request: AdConfigRequest): AdConfigResponse
}


@OptIn(InternalSerializationApi::class)
object AdConfigManager {
    private val retrofit: Retrofit by lazy {
        Retrofit.Builder()
            .baseUrl("https://msp-platform.newsbreak.com/")
            .addConverterFactory(GsonConverterFactory.create())
            .build()
    }
    private val api: AdConfigApi by lazy { retrofit.create(AdConfigApi::class.java) }

    private lateinit var appContext: Context
    private val jsonParser = Json { ignoreUnknownKeys = true }
    @Volatile private var cachedAdConfig: AdConfig? = null
    private const val configFileName = "msp_ad_config.json"
    private val externalAdConfigs: MutableMap<String, Placement> = mutableMapOf()

    private var _shouldLogMESEvent = false
    val shouldLogMESEvent: Boolean
        get() = _shouldLogMESEvent

    fun initialize(context: Context, initParams: InitializationParameters) {
        this.appContext = context.applicationContext

        cachedAdConfig = loadConfigFromLocalStorage()
        Logger.verbose("ad config loaded from local file: $cachedAdConfig")

        CoroutineScope(Dispatchers.Main).launch {
            Logger.info("sync ad config from remote server ...")
            fetchAdConfigFromServer(
                initParams.getAppId(),
                initParams.getOrgId(),
                initParams.getPrebidAPIKey()
            )
        }
    }

    private fun calculateShouldLogMESEvent(logConfig: LogConfig): Boolean {
        Logger.info("Calculate Should log MES event. logConfig: $logConfig")
        UserId.getCachedUserId()?.let {
            if (it.toString() in logConfig.msp_id_whitelist) {
                return true
            }
        }
        return Random.nextDouble() < logConfig.log_sample_rate
    }

    private suspend fun fetchAdConfigFromServer(
        appId: Int,
        orgId: Int,
        prebidApiKey: String
    ) = withContext(Dispatchers.IO) {
        try {
            val request = AdConfigRequest(
                app_id = appId,
                org_id = orgId,
                token = prebidApiKey
            )

            val response = api.fetchAdConfig(request)

            if (response.code == 0) {
                response.ad_config_settings?.let { settings ->
                    Logger.verbose("Ad config returned from remote server: ${settings.ad_config}")
                    val adConfig = jsonParser.decodeFromString<AdConfig>(settings.ad_config)
                    saveConfigToLocalStorage(adConfig)
                    cachedAdConfig = adConfig
                } ?: Logger.error("Ad Config settings are null.")
                response.log_config?.let {
                    _shouldLogMESEvent = calculateShouldLogMESEvent(it)
                    Logger.info("Should log MES event: $_shouldLogMESEvent")
                }
            } else {
                Logger.error("Failed to fetch ad config: ${response.message}")
            }
        } catch (e: Exception) {
            Logger.error("Exception occurred: ${e.message}")
        }
    }

    fun getAdConfig(placementId: String): Placement? {
        externalAdConfigs[placementId]?.let {
            // if Config of current placement has been set from Publisher App, just use it and ignore server config.
            Logger.verbose("Ad config returned from external cache: $it. MSP server config is ignored")
            return it
        }
        Logger.verbose("Ad config returned from server cache")
        return cachedAdConfig?.placements?.find { it.placementId == placementId } ?: defaultAdConfig(placementId)
    }

    fun setExternalAdConfigFromAdRequest(placement: String, adRequest: AdRequest) {
        (adRequest.customParams["msp_ad_config"] as? String)?.let {
            try {
                val adPlacementConfig = jsonParser.decodeFromString<Placement>(it)
                externalAdConfigs[placement] = adPlacementConfig
                Logger.verbose("Set Ad config from external: $it")
            } catch (e: Exception) {
                Logger.error("Exception occurred when parse external Ad config: ${e.message}")
            }
        }
    }

    internal fun getPlacementIds(): List<String> {
        return cachedAdConfig?.placements
            ?.map { placement ->
                placement.placementId
            }
            ?.filter { placementId ->
                placementId.contains("android", ignoreCase = true)
            }
            ?: emptyList()
    }

    private fun defaultAdConfig(placementId: String): Placement {
        return Placement(placementId, 8000, listOf(BidderInfo(name = "msp", bidderPlacementId = placementId)))
    }

    private fun saveConfigToLocalStorage(adConfig: AdConfig) {
        Logger.verbose("Save ad config from server to local file: $adConfig")
        val configJson = jsonParser.encodeToString(adConfig)
        appContext.openFileOutput(configFileName, Context.MODE_PRIVATE).use { it.write(configJson.toByteArray()) }
    }

    private fun loadConfigFromLocalStorage(): AdConfig? {
        return try {
            val configJson = appContext.openFileInput(configFileName).bufferedReader().use { it.readText() }
            jsonParser.decodeFromString(configJson)
        } catch (e: Exception) {
            null
        }
    }

    //private fun loadConfigFromRawResource(): AdConfig {
    //    val rawJson = loadJSONFromRaw(appContext, R.raw.local_ad_config)
    //    return jsonParser.decodeFromString(rawJson)
    //}
}

