package com.particles.msp.util

import android.Manifest
import android.app.ActivityManager
import android.content.Context
import android.content.pm.PackageManager
import android.content.res.Configuration
import android.hardware.display.DisplayManager
import android.net.ConnectivityManager
import android.os.BatteryManager
import android.os.Build
import android.os.PowerManager
import android.os.SystemClock
import android.view.Display
import android.view.Surface
import android.view.WindowManager
import androidx.annotation.RequiresPermission
import androidx.core.app.ActivityCompat
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.ProcessLifecycleOwner
import com.particles.mes.android.MesTracker
import com.particles.msp.api.MSPConstants.DEVICE_SIGNAL_BATTERY_LEVEL
import com.particles.msp.api.MSPConstants.DEVICE_SIGNAL_BATTERY_STATUS
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.json.JSONException
import org.json.JSONObject
import java.net.HttpURLConnection
import java.net.URL
import java.util.Locale
import java.util.TimeZone
import kotlin.math.abs

fun loadJSONFromRaw(context: Context, resourceId: Int): String {
    return context.resources.openRawResource(resourceId).bufferedReader().use { it.readText() }
}

fun getBidder(bidJsonStr: String): String {
    var bidder = ""
    try {
        bidder = JSONObject(bidJsonStr).getJSONObject("ext").getJSONObject("prebid").getJSONObject("meta")
            .getString("adaptercode")
    } catch (ignored: JSONException) {}
    return bidder
}

fun getBuckets(ext: Map<String, Any>?) : List<String> {
    val buckets = mutableListOf<String>()
    try {
        (ext?.get("msp_exp_bucket_info") as? JSONObject)?.optJSONArray("exp_bucket_list")?.let {
            for (i in 0 until it.length()) {
                it.optString(i)?.let { bucket ->
                    if (bucket.isNotEmpty()) {
                        buckets.add(bucket)
                    }
                }
            }
        }
    } catch (_: JSONException) {}
    return buckets
}

fun trackUrl(url: String) {
    CoroutineScope(Dispatchers.IO).launch {
        try {
            val connection = URL(url).openConnection() as HttpURLConnection
            connection.connect()
            connection.inputStream.close()
            connection.disconnect()
        } catch (_: Exception) {
        }
    }
}

fun getMesTracker(mesHostUrl: String): MesTracker? {
    return mesHostUrl
        .takeIf { it.isNotEmpty() }
        ?.let { MesTracker.obtain(it) }
}

fun getAppInstallTime(context: Context?): Long {
    val nonNullContext: Context = context ?: return 0

    return try {
        nonNullContext.packageManager.getPackageInfo(context.packageName, 0).firstInstallTime
    } catch (e: PackageManager.NameNotFoundException) {
        0
    }
}

fun getAvailableMemory(context: Context?): Long {
    val activityManager = context?.getSystemService(Context.ACTIVITY_SERVICE) as? ActivityManager

    if (activityManager == null) {
        Logger.info("getAvailableMemory: context is null or get ActivityManager failed.")
    }

    val memInfo = ActivityManager.MemoryInfo()
    activityManager?.getMemoryInfo(memInfo)
    return memInfo.availMem
}

fun getDeviceTimezone(): String {
    val timeZone = TimeZone.getDefault()
    val offsetInMillis = timeZone.getOffset(System.currentTimeMillis())

    val hours = offsetInMillis / 3_600_000
    val minutes = (offsetInMillis % 3_600_000) / 60_000

    return String.format(Locale.US, "%+03d:%02d", hours, abs(minutes))
}

fun getFontSize(context: Context?): String {
    return context?.let {
        val fontScale = it.resources.configuration.fontScale
        when {
            fontScale == 1.0f -> "m"
            fontScale < 1.0f -> "s"
            fontScale > 1.0f -> "l"
            else -> "unknown"
        }
    } ?: "unknown"
}

@RequiresPermission(Manifest.permission.ACCESS_NETWORK_STATE)
fun getIsLowDataMode(context: Context?): String {
    if (context == null) {
        Logger.info("getIsLowDataMode: context is null")
        return "unknown"
    }

    if (ActivityCompat.checkSelfPermission(
            context,
            Manifest.permission.ACCESS_NETWORK_STATE
        ) == PackageManager.PERMISSION_GRANTED
    ) {
        val connectivityManager =
            context.getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager

        if (connectivityManager == null) {
            Logger.info("getIsLowDataMode: get ConnectivityManager failed")
            return "unknown"
        } else {
            return if (connectivityManager.isActiveNetworkMetered) {
                "true"
            } else {
                "false"
            }
        }
    } else {
        Logger.info("getIsLowDataMode: no permission of ACCESS_NETWORK_STATE")
        return "unknown"
    }
}

fun getIsLowPowerMode(context: Context?): String {
    if (context == null) {
        Logger.info("getIsLowPowerMode: context is null")
        return "unknown"
    }

    val powerManager = context.getSystemService(Context.POWER_SERVICE) as? PowerManager
    if (powerManager == null) {
        Logger.info("getIsLowPowerMode: get PowerManager failed")
        return "unknown"
    } else {
        return if (powerManager.isPowerSaveMode) "true" else "false"
    }
}

fun setBatteryStatusAndLevel(
    context: Context?,
    customParams: MutableMap<String, Any>
) {
    if (context == null) {
        Logger.info("getBatteryStatus: context is null")
        customParams[DEVICE_SIGNAL_BATTERY_LEVEL] = "unknown"
        customParams[DEVICE_SIGNAL_BATTERY_STATUS] = "unknown"
    } else {
        val batteryManager = context.getSystemService(Context.BATTERY_SERVICE) as? BatteryManager
        if (batteryManager == null) {
            Logger.info("getBatteryStatus: get BatteryManager failed")
            customParams[DEVICE_SIGNAL_BATTERY_LEVEL] = "unknown"
            customParams[DEVICE_SIGNAL_BATTERY_STATUS] = "unknown"
        } else {
            val batteryCapacity =
                batteryManager.getIntProperty(BatteryManager.BATTERY_PROPERTY_CAPACITY)
            customParams[DEVICE_SIGNAL_BATTERY_LEVEL] = "%.2f".format(batteryCapacity / 100f)

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                val batteryStatus =
                    batteryManager.getIntProperty(BatteryManager.BATTERY_PROPERTY_STATUS)
                customParams[DEVICE_SIGNAL_BATTERY_STATUS] = when (batteryStatus) {
                    BatteryManager.BATTERY_STATUS_CHARGING -> "charging"
                    BatteryManager.BATTERY_STATUS_DISCHARGING -> "unplugged"
                    BatteryManager.BATTERY_STATUS_FULL -> "full"
                    BatteryManager.BATTERY_STATUS_NOT_CHARGING -> "unknown"
                    BatteryManager.BATTERY_STATUS_UNKNOWN -> "unknown"
                    else -> "unknown"
                }
            } else {
                customParams[DEVICE_SIGNAL_BATTERY_STATUS] = "unknown"
            }
        }
    }
}

fun getBatteryStatus(context: Context?): String {
    return if (context == null) {
        "unknown"
    } else {
        val batteryManager = context.getSystemService(Context.BATTERY_SERVICE) as? BatteryManager
        if (batteryManager == null) {
            "unknown"
        } else {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                val batteryStatus =
                    batteryManager.getIntProperty(BatteryManager.BATTERY_PROPERTY_STATUS)
                when (batteryStatus) {
                    BatteryManager.BATTERY_STATUS_CHARGING -> "charging"
                    BatteryManager.BATTERY_STATUS_DISCHARGING -> "unplugged"
                    BatteryManager.BATTERY_STATUS_FULL -> "full"
                    BatteryManager.BATTERY_STATUS_NOT_CHARGING -> "unknown"
                    BatteryManager.BATTERY_STATUS_UNKNOWN -> "unknown"
                    else -> "unknown"
                }
            } else {
                "unknown"
            }
        }
    }
}

fun getBatteryLevel(context: Context?): Float {
    return (context
        ?.getSystemService(Context.BATTERY_SERVICE) as? BatteryManager)
        ?.getIntProperty(BatteryManager.BATTERY_PROPERTY_CAPACITY)
        ?.div(100f)
        ?: 0f
}

fun getOrientation(context: Context?): String {
    if (context == null) {
        Logger.info("getOrientation: context is null")
        return "unknown"
    } else {

        val display = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            val displayManager =
                context.getSystemService(Context.DISPLAY_SERVICE) as? DisplayManager
            if (displayManager == null) {
                Logger.info("getOrientation: get DisplayManager failed")
                return "unknown"
            }

            displayManager.getDisplay(Display.DEFAULT_DISPLAY)
        } else {
            val windowManager = context.getSystemService(Context.WINDOW_SERVICE) as? WindowManager
            if (windowManager == null) {
                Logger.info("getOrientation: get WindowManager failed")
                return "unknown"
            }

            @Suppress("DEPRECATION")
            windowManager.defaultDisplay
        }

        val rotation = display.rotation
        val orientation = context.resources.configuration.orientation
        return when (rotation) {
            Surface.ROTATION_0 -> if (orientation == Configuration.ORIENTATION_PORTRAIT) "portrait" else "landscapeLeft"
            Surface.ROTATION_90 -> if (orientation == Configuration.ORIENTATION_LANDSCAPE) "landscapeLeft" else "portraitUpsideDown"
            Surface.ROTATION_180 -> if (orientation == Configuration.ORIENTATION_PORTRAIT) "portraitUpsideDown" else "landscapeRight"
            Surface.ROTATION_270 -> if (orientation == Configuration.ORIENTATION_LANDSCAPE) "landscapeRight" else "portrait"
            else -> "unknown"
        }
    }
}

fun getIsInForeground(): String {
    val isForeground = ProcessLifecycleOwner.get().lifecycle.currentState.isAtLeast(Lifecycle.State.STARTED)
    Logger.info("getIsInForeground: $isForeground")
    return if (isForeground) "true" else "false"
}

fun getLastSystemBootTime(): Long {
    return System.currentTimeMillis() - SystemClock.elapsedRealtime()
}

fun getLastSystemUpdateTime(context: Context?): Long {
    if (context == null) {
        Logger.info("getLastSystemUpdateTime: context is null")
        return 0L
    }

    return try {
        val pm = context.packageManager
        val info = pm.getPackageInfo("android", 0)
        info.lastUpdateTime
    } catch (e: PackageManager.NameNotFoundException) {
        Logger.info("getLastSystemUpdateTime failed: ${e.message}")
        0L
    }
}