package com.particles.msp.util

import android.Manifest
import android.content.Context
import androidx.annotation.RequiresPermission
import com.particles.mes.android.MesTracker
import com.particles.mes.android.data.MesAdClickEvent
import com.particles.mes.android.data.MesAdHideEvent
import com.particles.mes.android.data.MesAdImpressionEvent
import com.particles.mes.android.data.MesAdReportEvent
import com.particles.mes.android.data.MesAdRequest
import com.particles.mes.android.data.MesAdRequestEvent
import com.particles.mes.android.data.MesAdRequestExt
import com.particles.mes.android.data.MesAdResponse
import com.particles.mes.android.data.MesAdResponseEvent
import com.particles.mes.android.data.MesAdType
import com.particles.mes.android.data.MesGetAdEvent
import com.particles.mes.android.data.MesLoadAdEvent
import com.particles.mes.android.data.MesSdkInitEvent
import com.particles.mes.protos.ErrorCode
import com.particles.msp.MSPManager
import com.particles.msp.api.AdRequest
import com.particles.msp.api.BannerAdView
import com.particles.msp.api.MSPAd
import com.particles.msp.api.NativeAd
import com.particles.msp.auction.AdConfigManager
import org.json.JSONObject
import org.prebid.mobile.rendering.bidding.data.bid.Bid
import org.prebid.mobile.rendering.models.openrtb.BidRequest

object MesTrackerExt {
    private fun getMockedBid(ad: MSPAd): Bid {
        val bidJsonString = """
{
  "id": "",
  "impid": "",
  "adm": "",
  "adomain": [
  ],
  "crid": "",
  "ext": {
    "prebid": {
      "meta": {
        "adaptercode": "mockSeatPlaceholder"
      }
    }
  }
}
"""
        val bidJson = JSONObject(bidJsonString)
        bidJson.put("price", ad.adInfo["price"] as? Double ?: 0.0)
        bidJson.optJSONObject("ext")?.optJSONObject("prebid")?.optJSONObject("meta")
            ?.put("adaptercode", ad.adInfo["client_bidder"])
        return Bid.fromJSONObject(bidJson)
    }

    private fun getMesAdRequest(
        adRequest: AdRequest,
        winnerBid: Bid?,
        ad: MSPAd?,
        buckets: List<String>
    ): MesAdRequest {
        return MesAdRequest(
            sentRequestAtMillis = 0L,
            bidRequest = BidRequest().apply { id = winnerBid?.requestId ?: adRequest.uuid},
            ext = MesAdRequestExt(
                source = adRequest.placementId,
                position = adRequest.customParams["position"]?.toString()?.toIntOrNull() ?: 0,
                placementId = ad?.adInfo?.get("adnetwork_placement_id")?.toString()
                    ?: adRequest.placementId,
                buckets = buckets,
                adSlotId = adRequest.customParams["ad_slot_id"] as? String ?: "",
                userId = adRequest.customParams["user_id"] as? String ?: UserId.getCachedUserId()
                    ?.toString() ?: ""
            ),
            adRequest.org,
            adRequest.app,
            isOpenRtb = winnerBid != null
        )
    }

    private fun getMesAdResponse(ad: MSPAd, winnerBid: Bid?): MesAdResponse {
        return MesAdResponse(
            receivedResponseAtMillis = ad.adInfo["ad_loaded_at"] as? Long ?: 0L,
            adType = when (ad) {
                is BannerAdView -> MesAdType.AD_TYPE_DISPLAY
                is NativeAd -> MesAdType.AD_TYPE_NATIVE
                else -> MesAdType.AD_TYPE_UNSPECIFIED
            },
            bid = winnerBid ?: getMockedBid(ad),
            title = (ad as? NativeAd)?.title,
            body = (ad as? NativeAd)?.body,
            advertiser = (ad as? NativeAd)?.advertiser,
        )
    }

    fun MesTracker.trackAdImpression(
        adRequest: AdRequest,
        winnerBid: Bid?,
        ad: MSPAd,
        buckets: List<String>
    ) {
        this.trackAdImpression(
            MesAdImpressionEvent(
                adRequest = getMesAdRequest(adRequest, winnerBid, ad, buckets),
                adResponse = getMesAdResponse(ad, winnerBid),
                mspSdkVersion = MSPManager.version
            )
        )
    }

    fun MesTracker.trackAdClick(adRequest: AdRequest, winnerBid: Bid?, ad: MSPAd, buckets: List<String>) {
        this.trackAdClick(
            MesAdClickEvent(
                adRequest = getMesAdRequest(adRequest, winnerBid, ad, buckets),
                adResponse = getMesAdResponse(ad, winnerBid),
                mspSdkVersion = MSPManager.version
            )
        )
    }

    @RequiresPermission(Manifest.permission.ACCESS_NETWORK_STATE)
    fun MesTracker.trackSdkInit(
        org: String,
        app: String,
        latency: Int,
        totalCompleteTime: Int,
        completeTimeByAdNetwork: Map<String, Int>,
        mspId: String,
        context: Context,
        adId: String
    ) {
        this.trackSdkInit(
            MesSdkInitEvent(
                org = org,
                app = app,
                latency = latency,
                totalCompleteTime = totalCompleteTime,
                completeTimeByAdNetwork = completeTimeByAdNetwork,
                mspSdkVersion = MSPManager.version,
                mspId = mspId,
                lastSystemBootTimeMs = getLastSystemBootTime(),
                lastSystemUpdateTimeMs = getLastSystemUpdateTime(context),
                ifa = adId,
                batteryLevel = getBatteryLevel(context),
                batteryStatus = getBatteryStatus(context),
                fontSize = getFontSize(context),
                timeZone = getDeviceTimezone(),
                availableMemoryBytes = getAvailableMemory(context),
                isLowPowerMode = if (getIsLowPowerMode(context) == "true") true else false,
                isLowDataMode = if (getIsLowDataMode(context) == "true") true else false
            )
        )
    }

    fun MesTracker.trackLoadAd(
        adRequest: AdRequest,
        winnerBid: Bid?,
        ad: MSPAd?,
        buckets: List<String>,
        latency: Int,
        errorCode: ErrorCode,
        filledFromCache: Boolean = false,
        errorMessage: String? = null,
    ) {
        if (AdConfigManager.shouldLogMESEvent) {
            this.trackLoadAd(
                MesLoadAdEvent(
                    adRequest = getMesAdRequest(adRequest, winnerBid, ad, buckets),
                    adResponse = ad?.let { getMesAdResponse(ad, winnerBid) },
                    org = adRequest.org,
                    app = adRequest.app,
                    latency = latency,
                    errorCode = errorCode,
                    errorMessage = errorMessage,
                    filledFromCache = filledFromCache,
                    mspSdkVersion = MSPManager.version
                )
            )
        }
    }

    fun MesTracker.trackGetAd(
        adRequest: AdRequest?,
        winnerBid: Bid?,
        ad: MSPAd,
        buckets: List<String>,
        errorCode: ErrorCode,
        errorMessage: String? = null,
    ) {
        if (AdConfigManager.shouldLogMESEvent) {
            adRequest?.let {
                this.trackGetAd(
                    MesGetAdEvent(
                        adRequest = getMesAdRequest(it, winnerBid, ad, buckets),
                        adResponse = getMesAdResponse(ad, winnerBid),
                        errorCode = errorCode,
                        errorMessage = errorMessage,
                        mspSdkVersion = MSPManager.version
                    )
                )
            }
        }
    }

    fun MesTracker.trackAdHide(
        adRequest: AdRequest?, winnerBid: Bid?, ad: MSPAd,
        reason: String,
        adScreenshot: ByteArray? = null,
        fullScreenshot: ByteArray? = null,
        buckets: List<String>
    ) {
        adRequest?.let {
            this.trackAdHide(
                MesAdHideEvent(
                    adRequest = getMesAdRequest(it, winnerBid, ad, buckets),
                    adResponse = getMesAdResponse(ad, winnerBid).copy(
                        adScreenshot = adScreenshot,
                        fullScreenshot = fullScreenshot,
                    ),
                    reason = reason,
                    mspSdkVersion = MSPManager.version
                )
            )
        }
    }

    fun MesTracker.trackAdReport(
        adRequest: AdRequest?, winnerBid: Bid?, ad: MSPAd,
        reason: String, description: String?,
        adScreenshot: ByteArray? = null,
        fullScreenshot: ByteArray? = null,
        buckets: List<String>
    ) {
        adRequest?.let {
            this.trackAdReport(
                MesAdReportEvent(
                    adRequest = getMesAdRequest(it, winnerBid, ad, buckets),
                    adResponse = getMesAdResponse(ad, winnerBid).copy(
                        adScreenshot = adScreenshot,
                        fullScreenshot = fullScreenshot,
                    ),
                    reason = reason,
                    description = description,
                    mspSdkVersion = MSPManager.version
                )
            )
        }
    }

    fun MesTracker.trackAdRequest(
        adRequest: AdRequest,
        winnerBid: Bid? = null,
    ) {
        if (AdConfigManager.shouldLogMESEvent) {
            this.trackAdRequest(
                MesAdRequestEvent(
                    adRequest = getMesAdRequest(
                        adRequest,
                        winnerBid,
                        null,
                        getBuckets(null),
                    ),
                    mspSdkVersion = MSPManager.version
                )
            )
        }
    }

    fun MesTracker.trackAdResponse(
        adRequest: AdRequest,
        winnerBid: Bid?,
        ad: MSPAd?,
        buckets: List<String>,
        latency: Int,
        errorCode: ErrorCode,
        errorMessage: String? = null,
    ) {
        if (AdConfigManager.shouldLogMESEvent) {
            this.trackAdResponse(
                MesAdResponseEvent(
                    adRequest = getMesAdRequest(adRequest, winnerBid, ad, buckets),
                    adResponse = ad?.let { getMesAdResponse(it, winnerBid) },
                    latency = latency,
                    errorCode = errorCode,
                    errorMessage = errorMessage,
                    mspSdkVersion = MSPManager.version
                )
            )
        }
    }
}