package com.particles.msp.api

import com.particles.mes.protos.ErrorCode
import com.particles.mes.protos.GetAdEvent
import com.particles.mes.protos.RequestContext
import com.particles.mes.protos.RequestContextExt
import com.particles.msp.AdCache
import com.particles.msp.MSPManager
import com.particles.msp.auction.AdConfigManager
import com.particles.msp.auction.Auction
import com.particles.msp.auction.AuctionBid
import com.particles.msp.auction.AuctionListener
import com.particles.msp.util.Logger
import com.particles.msp.util.MesTrackerExt.trackLoadAd
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import org.prebid.mobile.rendering.bidding.data.bid.BidResponse
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

class AdLoader {
    fun loadAd(placementId: String, adListener: AdListener, adRequest: AdRequest) {
        val adLoadStartTime = System.currentTimeMillis()
        val biddersInfo = AdConfigManager.getAdConfig(placementId)
        val bidderList = biddersInfo?.bidders?.mapNotNull { MSPManager.bidderProvider?.getBidder(it) } ?: emptyList()
        Logger.info("Starting auction for placementId: $placementId. requestId: ${adRequest.uuid}")
        Auction(bidderList, adRequest, false, biddersInfo?.auctionTimeout ?: 8000).startAuction(object : AuctionListener {
            override fun onSuccess(winningBid: AuctionBid) {
                val ad = AdCache.peakAd(winningBid.bidderPlacementId)
                val latency = (System.currentTimeMillis() - adLoadStartTime).toInt()
                sendLoadAdEvent(
                    latency = latency,
                    adRequest = adRequest,
                    ad = ad,
                    fromCache = winningBid.fromCache
                )
                CoroutineScope(Dispatchers.Main).launch {
                    adListener.onAdLoaded(adRequest.placementId)
                }
            }

            override fun onError(error: String) {
                val latency = (System.currentTimeMillis() - adLoadStartTime).toInt()
                sendLoadAdEvent(
                    latency = latency,
                    adRequest = adRequest,
                    errorMessage = error
                )
                CoroutineScope(Dispatchers.Main).launch {
                    adListener.onError("No winning bid from auction: $error")
                }
            }

        }, adListener)
    }

    fun getAd(placementId: String): MSPAd? {
        val winnerAuctionBid = runBlocking { getWinnerBidFromCache(placementId) }
        val ad = winnerAuctionBid?.let {
            AdCache.getAd(it.bidderPlacementId)
        }

        if (ad == null) {
            val errorMessage = "[AdLoader: GetAd] failed to get ad from cache"
            Logger.verbose(errorMessage)
            trackGetAdFailed(placementId, errorMessage)
        } else {
            ad.trackGetAdSuccessEvent()
        }

        return ad
    }

    private suspend fun getWinnerBidFromCache(placementId: String): AuctionBid? = suspendCoroutine { continuation ->
        var isResumed = false
        fun safeResume(value: AuctionBid?) {
            if (!isResumed) {
                isResumed = true
                continuation.resume(value)
            } else {
                Logger.warning("[AdLoader: getAd] Attempted to resume continuation more than once. Ignoring.")
            }
        }

        val biddersInfo = AdConfigManager.getAdConfig(placementId)
        val bidderList = biddersInfo?.bidders?.mapNotNull { MSPManager.bidderProvider?.getBidder(it) } ?: emptyList()
        Auction(bidderList, null, true, 8000).startAuction(object : AuctionListener {
            override fun onSuccess(winningBid: AuctionBid) {
                safeResume(winningBid) // Resume coroutine with the result
            }

            override fun onError(error: String) {
                safeResume(null) // Resume coroutine with the result
            }

        }, null)
    }

    private fun sendLoadAdEvent(
        latency: Int,
        adRequest: AdRequest,
        ad: MSPAd? = null,
        bidResponse: BidResponse? = null,
        fromCache: Boolean = false,
        errorCode: ErrorCode = ErrorCode.ERROR_CODE_NO_FILL,
        errorMessage: String? = null
    ) {
        if (ad != null) {
            ad.trackLoadAdSuccessEvent(latency, fromCache)
        } else {
            MSPManager.mesTracker?.trackLoadAd(
                adRequest,
                bidResponse,
                ad,
                latency,
                errorCode,
                fromCache,
                errorMessage
            )
        }
    }

    private fun trackGetAdFailed(
        placementId: String,
        errorMessage: String,
    ) {
        if (!AdConfigManager.shouldLogMESEvent) return

        val requestContextExt = RequestContextExt.newBuilder()
            .apply {
                this.placementId = placementId
                this.userId = MSPManager.mesMspUserId
            }
            .build()

        val requestContext = RequestContext.newBuilder()
            .apply {
                tsMs = 0L
                ext = requestContextExt
            }
            .build()

        MSPManager.mesTracker?.trackGetAd(
            GetAdEvent.newBuilder().apply {
                this.org = MSPManager.org
                this.app = MSPManager.app
                this.requestContext = requestContext
                this.errorCode = ErrorCode.ERROR_CODE_NO_FILL
                this.errorMessage = errorMessage
                this.mspSdkVersion = MSPManager.version
            }
            .build()
        )
    }
}
