package com.particles.msp.util

import android.Manifest
import android.app.ActivityManager
import android.content.Context
import android.content.pm.PackageManager
import android.content.res.Configuration
import android.hardware.display.DisplayManager
import android.media.AudioManager
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.os.BatteryManager
import android.os.Build
import android.os.Environment
import android.os.PowerManager
import android.os.StatFs
import android.provider.Settings
import android.telephony.TelephonyManager
import android.view.Display
import android.view.Surface
import android.view.WindowManager
import androidx.annotation.RequiresPermission
import androidx.appcompat.app.AppCompatDelegate
import androidx.core.app.ActivityCompat
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.ProcessLifecycleOwner
import com.particles.mes.android.MesTracker
import com.particles.mes.android.data.AppSignal
import com.particles.mes.android.data.ConnectionType
import com.particles.mes.android.data.DeviceSignal
import com.particles.mes.android.data.SdkSignal
import com.particles.mes.protos.SdkPlatform
import com.particles.msp.MSPManager
import com.particles.msp.MSPManager.app
import com.particles.msp.MSPManager.org
import com.particles.msp.MSPManager.version
import com.particles.msp.api.MSPConstants.DEVICE_SIGNAL_BATTERY_LEVEL
import com.particles.msp.api.MSPConstants.DEVICE_SIGNAL_BATTERY_STATUS
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.json.JSONException
import org.json.JSONObject
import org.prebid.mobile.rendering.sdk.ManagersResolver
import org.prebid.mobile.rendering.sdk.deviceData.managers.DeviceInfoManager
import org.prebid.mobile.rendering.utils.helpers.AdIdManager
import org.prebid.mobile.rendering.utils.helpers.AppInfoManager
import java.net.HttpURLConnection
import java.net.URL
import java.util.Locale
import java.util.TimeZone
import kotlin.math.abs

fun loadJSONFromRaw(context: Context, resourceId: Int): String {
    return context.resources.openRawResource(resourceId).bufferedReader().use { it.readText() }
}

fun getBidder(bidJsonStr: String): String {
    var bidder = ""
    try {
        bidder = JSONObject(bidJsonStr).getJSONObject("ext").getJSONObject("prebid").getJSONObject("meta")
            .getString("adaptercode")
    } catch (ignored: JSONException) {}
    return bidder
}

fun getBuckets(ext: Map<String, Any>?) : List<String> {
    val buckets = mutableListOf<String>()
    try {
        (ext?.get("msp_exp_bucket_info") as? JSONObject)?.optJSONArray("exp_bucket_list")?.let {
            for (i in 0 until it.length()) {
                it.optString(i)?.let { bucket ->
                    if (bucket.isNotEmpty()) {
                        buckets.add(bucket)
                    }
                }
            }
        }
    } catch (_: JSONException) {}
    return buckets
}

fun getInferredCountry(ext: Map<String, Any>?): String? {
    return ext?.get("inferred_country") as? String
}

fun trackUrl(url: String) {
    CoroutineScope(Dispatchers.IO).launch {
        try {
            val connection = URL(url).openConnection() as HttpURLConnection
            connection.connect()
            connection.inputStream.close()
            connection.disconnect()
        } catch (_: Exception) {
        }
    }
}

fun getMesTracker(mesHostUrl: String): MesTracker? {
    return mesHostUrl
        .takeIf { it.isNotEmpty() }
        ?.let { MesTracker.obtain(it) }
}

fun getAppInstallTime(context: Context?): Long {
    val nonNullContext: Context = context ?: return 0

    return try {
        nonNullContext.packageManager.getPackageInfo(context.packageName, 0).firstInstallTime
    } catch (e: PackageManager.NameNotFoundException) {
        0
    }
}

fun getAvailableMemory(context: Context?): Long {
    val activityManager = context?.getSystemService(Context.ACTIVITY_SERVICE) as? ActivityManager

    if (activityManager == null) {
        Logger.info("getAvailableMemory: context is null or get ActivityManager failed.")
    }

    val memInfo = ActivityManager.MemoryInfo()
    activityManager?.getMemoryInfo(memInfo)
    return memInfo.availMem
}

fun getTotalMemory(context: Context): Long {
    val activityManager = context.getSystemService(Context.ACTIVITY_SERVICE) as? ActivityManager

    if (activityManager == null) {
        Logger.info("getAvailableMemory: context is null or get ActivityManager failed.")
    }

    val memInfo = ActivityManager.MemoryInfo()
    activityManager?.getMemoryInfo(memInfo)
    return memInfo.totalMem
}

data class StorageStats(
    val totalStorageBytes: Long,
    val availableStorageBytes: Long
)

fun getStorageStats(): StorageStats {
    val path = Environment.getDataDirectory()   // Internal storage (/data)
    val stat = StatFs(path.path)

    val blockSize = stat.blockSizeLong
    val totalBlocks = stat.blockCountLong
    val availableBlocks = stat.availableBlocksLong

    return StorageStats(
        totalStorageBytes = totalBlocks * blockSize,
        availableStorageBytes = availableBlocks * blockSize
    )
}

fun getDeviceTimezone(): String {
    val timeZone = TimeZone.getDefault()
    val offsetInMillis = timeZone.getOffset(System.currentTimeMillis())

    val hours = offsetInMillis / 3_600_000
    val minutes = (offsetInMillis % 3_600_000) / 60_000

    return String.format(Locale.US, "%+03d:%02d", hours, abs(minutes))
}

fun getFontSize(context: Context?): String {
    return context?.let {
        val fontScale = it.resources.configuration.fontScale
        when {
            fontScale == 1.0f -> "m"
            fontScale < 1.0f -> "s"
            fontScale > 1.0f -> "l"
            else -> "unknown"
        }
    } ?: "unknown"
}

@RequiresPermission(Manifest.permission.ACCESS_NETWORK_STATE)
fun getIsLowDataMode(context: Context?): String {
    if (context == null) {
        Logger.info("getIsLowDataMode: context is null")
        return "unknown"
    }

    if (ActivityCompat.checkSelfPermission(
            context,
            Manifest.permission.ACCESS_NETWORK_STATE
        ) == PackageManager.PERMISSION_GRANTED
    ) {
        val connectivityManager =
            context.getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager

        if (connectivityManager == null) {
            Logger.info("getIsLowDataMode: get ConnectivityManager failed")
            return "unknown"
        } else {
            return if (connectivityManager.isActiveNetworkMetered) {
                "true"
            } else {
                "false"
            }
        }
    } else {
        Logger.info("getIsLowDataMode: no permission of ACCESS_NETWORK_STATE")
        return "unknown"
    }
}

fun getIsLowPowerMode(context: Context?): String {
    if (context == null) {
        Logger.info("getIsLowPowerMode: context is null")
        return "unknown"
    }

    val powerManager = context.getSystemService(Context.POWER_SERVICE) as? PowerManager
    if (powerManager == null) {
        Logger.info("getIsLowPowerMode: get PowerManager failed")
        return "unknown"
    } else {
        return if (powerManager.isPowerSaveMode) "true" else "false"
    }
}

fun setBatteryStatusAndLevel(
    context: Context?,
    customParams: MutableMap<String, Any>
) {
    if (context == null) {
        Logger.info("getBatteryStatus: context is null")
        customParams[DEVICE_SIGNAL_BATTERY_LEVEL] = "unknown"
        customParams[DEVICE_SIGNAL_BATTERY_STATUS] = "unknown"
    } else {
        val batteryManager = context.getSystemService(Context.BATTERY_SERVICE) as? BatteryManager
        if (batteryManager == null) {
            Logger.info("getBatteryStatus: get BatteryManager failed")
            customParams[DEVICE_SIGNAL_BATTERY_LEVEL] = "unknown"
            customParams[DEVICE_SIGNAL_BATTERY_STATUS] = "unknown"
        } else {
            val batteryCapacity =
                batteryManager.getIntProperty(BatteryManager.BATTERY_PROPERTY_CAPACITY)
            customParams[DEVICE_SIGNAL_BATTERY_LEVEL] = "%.2f".format(batteryCapacity / 100f)

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                val batteryStatus =
                    batteryManager.getIntProperty(BatteryManager.BATTERY_PROPERTY_STATUS)
                customParams[DEVICE_SIGNAL_BATTERY_STATUS] = when (batteryStatus) {
                    BatteryManager.BATTERY_STATUS_CHARGING -> "charging"
                    BatteryManager.BATTERY_STATUS_DISCHARGING -> "unplugged"
                    BatteryManager.BATTERY_STATUS_FULL -> "full"
                    BatteryManager.BATTERY_STATUS_NOT_CHARGING -> "unknown"
                    BatteryManager.BATTERY_STATUS_UNKNOWN -> "unknown"
                    else -> "unknown"
                }
            } else {
                customParams[DEVICE_SIGNAL_BATTERY_STATUS] = "unknown"
            }
        }
    }
}

fun getBatteryStatus(context: Context?): String {
    return if (context == null) {
        "unknown"
    } else {
        val batteryManager = context.getSystemService(Context.BATTERY_SERVICE) as? BatteryManager
        if (batteryManager == null) {
            "unknown"
        } else {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                val batteryStatus =
                    batteryManager.getIntProperty(BatteryManager.BATTERY_PROPERTY_STATUS)
                when (batteryStatus) {
                    BatteryManager.BATTERY_STATUS_CHARGING -> "charging"
                    BatteryManager.BATTERY_STATUS_DISCHARGING -> "unplugged"
                    BatteryManager.BATTERY_STATUS_FULL -> "full"
                    BatteryManager.BATTERY_STATUS_NOT_CHARGING -> "unknown"
                    BatteryManager.BATTERY_STATUS_UNKNOWN -> "unknown"
                    else -> "unknown"
                }
            } else {
                "unknown"
            }
        }
    }
}

fun getThermalStatus(context: Context): String {
    val powerManager = context.getSystemService(Context.POWER_SERVICE) as PowerManager
    return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
        when (powerManager.currentThermalStatus) {
            PowerManager.THERMAL_STATUS_NONE -> "none"                // Normal
            PowerManager.THERMAL_STATUS_LIGHT -> "light"              // Slightly warm
            PowerManager.THERMAL_STATUS_MODERATE -> "moderate"        // Getting warm
            PowerManager.THERMAL_STATUS_SEVERE -> "severe"            // May throttle CPU/GPU
            PowerManager.THERMAL_STATUS_CRITICAL -> "critical"        // Major throttling
            PowerManager.THERMAL_STATUS_EMERGENCY -> "emergency"      // Shutdown imminent
            PowerManager.THERMAL_STATUS_SHUTDOWN -> "shutdown"        // Thermal shutdown
            else -> "unknown"
        }
    } else {
        "unknown"
    }
}

fun getThemeStatus(context: Context): String {
    val appMode = AppCompatDelegate.getDefaultNightMode()
    val currentMode = context.resources.configuration.uiMode and Configuration.UI_MODE_NIGHT_MASK

    return when {
        appMode == AppCompatDelegate.MODE_NIGHT_FOLLOW_SYSTEM -> "system"
        currentMode == Configuration.UI_MODE_NIGHT_YES -> "dark"
        currentMode == Configuration.UI_MODE_NIGHT_NO -> "light"
        else -> "unknown"
    }
}

fun getBatteryLevel(context: Context?): Float {
    return (context
        ?.getSystemService(Context.BATTERY_SERVICE) as? BatteryManager)
        ?.getIntProperty(BatteryManager.BATTERY_PROPERTY_CAPACITY)
        ?.div(100f)
        ?: 0f
}

fun getBatteryLevelInt(context: Context): Int {
    return (context.getSystemService(Context.BATTERY_SERVICE) as? BatteryManager)
        ?.getIntProperty(BatteryManager.BATTERY_PROPERTY_CAPACITY) ?: -1
}

fun getVolumeLevel(context: Context): Int {
    val audioManager = context.getSystemService(Context.AUDIO_SERVICE) as AudioManager
    return audioManager.getStreamVolume(AudioManager.STREAM_MUSIC)
}

fun getSystemBrightness(context: Context): Int {
    return try {
        Settings.System.getInt(
            context.contentResolver,
            Settings.System.SCREEN_BRIGHTNESS
        )
    } catch (e: Settings.SettingNotFoundException) {
        e.printStackTrace()
        -1
    }
}

fun getOrientation(context: Context?): String {
    if (context == null) {
        Logger.info("getOrientation: context is null")
        return "unknown"
    } else {

        val display = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            val displayManager =
                context.getSystemService(Context.DISPLAY_SERVICE) as? DisplayManager
            if (displayManager == null) {
                Logger.info("getOrientation: get DisplayManager failed")
                return "unknown"
            }

            displayManager.getDisplay(Display.DEFAULT_DISPLAY)
        } else {
            val windowManager = context.getSystemService(Context.WINDOW_SERVICE) as? WindowManager
            if (windowManager == null) {
                Logger.info("getOrientation: get WindowManager failed")
                return "unknown"
            }

            @Suppress("DEPRECATION")
            windowManager.defaultDisplay
        }

        val rotation = display.rotation
        val orientation = context.resources.configuration.orientation
        return when (rotation) {
            Surface.ROTATION_0 -> if (orientation == Configuration.ORIENTATION_PORTRAIT) "portrait" else "landscapeLeft"
            Surface.ROTATION_90 -> if (orientation == Configuration.ORIENTATION_LANDSCAPE) "landscapeLeft" else "portraitUpsideDown"
            Surface.ROTATION_180 -> if (orientation == Configuration.ORIENTATION_PORTRAIT) "portraitUpsideDown" else "landscapeRight"
            Surface.ROTATION_270 -> if (orientation == Configuration.ORIENTATION_LANDSCAPE) "landscapeRight" else "portrait"
            else -> "unknown"
        }
    }
}

fun getIsInForeground(): String {
    val isForeground = ProcessLifecycleOwner.get().lifecycle.currentState.isAtLeast(Lifecycle.State.STARTED)
    Logger.info("getIsInForeground: $isForeground")
    return if (isForeground) "true" else "false"
}

fun getCellularConnectionType(context: Context): ConnectionType {
    val telephonyManager =
        context.getSystemService(Context.TELEPHONY_SERVICE) as TelephonyManager

    return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
        when (telephonyManager.dataNetworkType) {
            TelephonyManager.NETWORK_TYPE_GPRS,
            TelephonyManager.NETWORK_TYPE_EDGE,
            TelephonyManager.NETWORK_TYPE_CDMA,
            TelephonyManager.NETWORK_TYPE_1xRTT,
            TelephonyManager.NETWORK_TYPE_IDEN ->
                ConnectionType.CONNECTION_TYPE_CELL_2G

            TelephonyManager.NETWORK_TYPE_UMTS,
            TelephonyManager.NETWORK_TYPE_EVDO_0,
            TelephonyManager.NETWORK_TYPE_EVDO_A,
            TelephonyManager.NETWORK_TYPE_HSDPA,
            TelephonyManager.NETWORK_TYPE_HSUPA,
            TelephonyManager.NETWORK_TYPE_HSPA,
            TelephonyManager.NETWORK_TYPE_EVDO_B,
            TelephonyManager.NETWORK_TYPE_EHRPD,
            TelephonyManager.NETWORK_TYPE_HSPAP ->
                ConnectionType.CONNECTION_TYPE_CELL_3G

            TelephonyManager.NETWORK_TYPE_LTE ->
                ConnectionType.CONNECTION_TYPE_CELL_4G

            TelephonyManager.NETWORK_TYPE_NR -> // 5G New Radio
                ConnectionType.CONNECTION_TYPE_CELL_5G

            else ->
                ConnectionType.CONNECTION_TYPE_CELL_UNKNOWN
        }
    } else {
        ConnectionType.CONNECTION_TYPE_CELL_UNKNOWN
    }
}

fun getConnectionType(context: Context): ConnectionType {
    val connectivityManager =
        context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager

    val network = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
        connectivityManager.activeNetwork ?: return ConnectionType.CONNECTION_TYPE_UNKNOWN
    } else {
        return ConnectionType.CONNECTION_TYPE_UNKNOWN
    }
    val capabilities = connectivityManager.getNetworkCapabilities(network) ?: return ConnectionType.CONNECTION_TYPE_UNKNOWN

    return when {
        capabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI) ->
            ConnectionType.CONNECTION_TYPE_WIFI
        capabilities.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET) ->
            ConnectionType.CONNECTION_TYPE_ETHERNET
        capabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR) ->
            getCellularConnectionType(context)
        else ->
            ConnectionType.CONNECTION_TYPE_UNKNOWN
    }
}

fun getCountry(): String {
    return Locale.getDefault().country
}

fun getLocale(): String {
    return Locale.getDefault().toLanguageTag()
}

fun getTimezone(): String {
    return TimeZone.getDefault().id
}

fun getSDKSignal(): SdkSignal {
    return SdkSignal(
        orgId = org.toIntOrNull() ?: -1,
        appId = app.toIntOrNull() ?: -1,
        mspId = UserId.getCachedUserId(),
        sdkVersion = version,
        platform = SdkPlatform.SDK_PLATFORM_ANDROID
    )
}

fun getAppSignal(): AppSignal {
    return AppSignal(
        name = AppInfoManager.getAppName() ?: "",
        bundle = AppInfoManager.getPackageName() ?: "",
        ver = AppInfoManager.getAppVersion() ?: "",
        domain = "",
        page = "",
        ppid = MSPManager.ppid
    )
}

suspend fun getDeviceSignal(context: Context): DeviceSignal {
    val storageStats = getStorageStats()
    val deviceManager: DeviceInfoManager? = ManagersResolver.getInstance().deviceManager;
    return DeviceSignal(
        make = Build.MANUFACTURER,
        model = Build.MODEL,
        os = "Android",
        osv = Build.VERSION.RELEASE,
        kernelv = System.getProperty("os.version") ?: "unknown",
        w = deviceManager?.screenWidth ?: 0,
        h = deviceManager?.screenHeight ?: 0,
        orientation = getOrientation(context),
        volumeLevel = getVolumeLevel(context),
        brightness = getSystemBrightness(context),
        batteryLevel = getBatteryLevelInt(context),
        batteryStatus = getBatteryStatus(context),
        thermalStatus = getThermalStatus(context),
        totalMemoryBytes = getTotalMemory(context),
        availableMemoryBytes = getAvailableMemory(context),
        totalStorageBytes = storageStats.totalStorageBytes,
        availableStorageBytes = storageStats.availableStorageBytes,
        isLowPowerMode = getIsLowPowerMode(context) == "true",
        isLowDataMode = getIsLowDataMode(context) == "true",
        fontSize = getFontSize(context),
        theme = getThemeStatus(context),
        carrier = deviceManager?.carrier ?: "unknown",
        mccmnc = deviceManager?.mccMnc ?: "unknown",
        connectionType = getConnectionType(context),
        country = getCountry(),
        timezone = getTimezone(),
        locale = getLocale(),
        ua = AppInfoManager.getUserAgent(),
        ifa = AdIdRepo.get(context),
        ifv = AppSetIdRepo.get(context),
        lmt = AdIdManager.isLimitAdTrackingEnabled(),
    )
}