package com.particles.novaadapter

import android.app.Activity
import android.content.Context
import com.particles.android.ads.AdError
import com.particles.android.ads.AdLoadListener
import com.particles.android.ads.AdLoader
import com.particles.android.ads.AdRequest
import com.particles.android.ads.AdSize
import com.particles.android.ads.NovaSdk
import com.particles.android.ads.banner.BannerAd
import com.particles.android.ads.banner.BannerAdListener
import com.particles.android.ads.interstitial.InterstitialAd
import com.particles.android.ads.interstitial.InterstitialAdListener
import com.particles.android.ads.nativead.MediaView
import com.particles.android.ads.nativead.NativeAd
import com.particles.android.ads.nativead.NativeAdListener
import com.particles.android.ads.nativead.NativeAdView
import com.particles.mes.android.MesTracker
import com.particles.mes.android.MesTrackerExt.trackAdImpression
import com.particles.msp.adapter.AdNetwork
import com.particles.msp.adapter.AdNetworkAdapter
import com.particles.msp.adapter.AdapterInitListener
import com.particles.msp.adapter.AdapterInitStatus
import com.particles.msp.adapter.InitializationParameters
import com.particles.msp.util.Logger
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.json.JSONException
import org.json.JSONObject
import org.prebid.mobile.rendering.bidding.data.bid.Bid
import org.prebid.mobile.rendering.bidding.data.bid.BidResponse
import com.particles.msp.api.AdListener as MspAdListener
import com.particles.msp.api.AdRequest as MspAdRequest
import com.particles.msp.api.BannerAdView as MspBannerAdView
import com.particles.msp.api.NativeAd as MspNativeAd
import com.particles.msp.api.NativeAdView as MspNativeAdView

class NovaAdapter : AdNetworkAdapter() {
    private var nativeAd: NativeAd? = null
    private var bannerAd: BannerAd? = null
    private var interstitialAd: InterstitialAd? = null

    override fun destroyAd() {
        nativeAd?.destroy()
        nativeAd = null
        bannerAd = null
        interstitialAd = null
    }

    override fun initialize(
        initParams: InitializationParameters,
        adapterInitListener: AdapterInitListener,
        context: Any?
    ) {
        mesTracker = initParams.getMesHostUrl().takeIf { it.isNotEmpty() }
            ?.let { MesTracker.obtain(it) }
        CoroutineScope(Dispatchers.IO).launch {
            NovaSdk.initialize(context as Context, initParams.getNovaServerApiHost()) {
                adapterInitListener.onComplete(
                    AdNetwork.Nova,
                    AdapterInitStatus.SUCCESS,
                    AdapterInitStatus.SUCCESS.message
                )
            }
        }
    }

    override fun loadAdCreative(
        bidResponse: Any,
        adListener: MspAdListener,
        context: Any,
        adRequest: MspAdRequest
    ) {
        if (bidResponse !is BidResponse || context !is Context) {
            adListener.onError("Failed to load Nova Ad: bidResponse or context is missing")
            return
        }

        val winnerBid = bidResponse.winningBid
        if (winnerBid == null) {
            adListener.onError("Failed to load Nova Ad: No Winning bid returned")
            return
        }

        val adUnitId = getOriginalNovaAdUnitId(winnerBid)
        if (adUnitId == null) {
            adListener.onError("Failed to load Nova Ad: failed to parse original ad unit id")
            return
        }

        if (adRequest.adFormat == com.particles.msp.api.AdFormat.INTERSTITIAL) {
            loadInterstitialAd(context, adUnitId, winnerBid, adRequest, adListener)
            return
        }

        when (val type = winnerBid.prebid.type) {
            "banner" -> loadBannerAd(context, adUnitId, winnerBid, adRequest, adListener)
            "native" -> loadNativeAd(context, adUnitId, winnerBid, adRequest, adListener)
            else -> adListener.onError("Failed to load Nova Ad: Unsupported Ad type: $type")
        }
    }

    private fun loadBannerAd(
        context: Context,
        adUnitId: String,
        winnerBid: Bid,
        adRequest: MspAdRequest,
        adListener: MspAdListener
    ) {
        val adLoadListener = object : AdLoadListener<BannerAd> {
            override fun onAdLoaded(ads: List<BannerAd>) {
                Logger.info("successfully loaded nova banner ads ...")
                val ad = ads[0].also { bannerAd = it }

                val mspAd = MspBannerAdView(ad.getAdView(), this@NovaAdapter)
                    .apply {
                        adInfo["price"] = winnerBid.price
                        adInfo["bidder_placement_id"] = adUnitId
                        adInfo["ad_loaded_at"] = System.currentTimeMillis()
                    }

                ad.setAdListener(object : BannerAdListener {
                    override fun onAdImpressed() {
                        mesTracker?.trackAdImpression(adRequest, winnerBid, mspAd)
                        handleAdImpression(adListener, adRequest)
                    }
                    override fun onAdClicked() {
                        handleAdClicked(adListener, adRequest)
                    }
                })

                handleAdLoaded(mspAd, adListener, adRequest)
            }

            override fun onAdLoadFailed(error: AdError) {
                Logger.info("failed to load nova banner ads ...${error.message}")
                adListener.onError("Failed to load Nova Ad: ${error.message}")
            }
        }

        val adLoader = AdLoader.Builder(context, adUnitId)
            .forBannerAd(adLoadListener)
            .build()

        adLoader?.loadAd(
            AdRequest.Builder(winnerBid.impId)
                .setAdSize(adRequest.adSize?.let { AdSize(it.width, it.height) })
                .setAdString(winnerBid.adm)
                .build()
        )
    }

    private fun loadInterstitialAd(
        context: Context,
        adUnitId: String,
        winnerBid: Bid,
        adRequest: MspAdRequest,
        adListener: MspAdListener
    ) {
        val adLoadListener = object : AdLoadListener<InterstitialAd>  {
            private fun onAdLoaded(interstitialAd: InterstitialAd) {
                val mspInterstitialAd = NovaInterstitialAd(this@NovaAdapter, interstitialAd)
                interstitialAd.setAdListener(object : InterstitialAdListener {
                    override fun onAdImpressed() {
                        mesTracker?.trackAdImpression(adRequest, winnerBid, mspInterstitialAd)
                        handleAdImpression(adListener, adRequest)
                    }

                    override fun onAdClicked() {
                        handleAdClicked(adListener, adRequest)
                    }

                    override fun onAdDismissed() {
                        handleAdDismissed(adListener, adRequest)
                    }
                })
                mspInterstitialAd.adInfo["price"] = winnerBid.price
                mspInterstitialAd.adInfo["bidder_placement_id"] = adUnitId
                mspInterstitialAd.adInfo["ad_loaded_at"] = System.currentTimeMillis()
                handleAdLoaded(mspInterstitialAd, adListener, adRequest)
            }

            override fun onAdLoaded(ads: List<InterstitialAd>) {
                Logger.info("successfully loaded nova interstitial ads ...")
                val ad = ads[0].also { interstitialAd = it }
                onAdLoaded(ad)
            }

            override fun onAdLoadFailed(error: AdError) {
                adListener.onError("Failed to load Nova native Ad: ${error.message}")
            }
        }

        val adLoader = AdLoader.Builder(context, adUnitId)
            .forInterstitialAd(adLoadListener)
            .build()

        adLoader?.loadAd(
            AdRequest.Builder(winnerBid.impId)
                .setAdSize(adRequest.adSize?.let { AdSize(it.width, it.height) })
                .setAdString(winnerBid.adm)
                .build()
        )
    }

    private fun loadNativeAd(
        context: Context,
        adUnitId: String,
        winnerBid: Bid,
        adRequest: MspAdRequest,
        adListener: MspAdListener
    ) {
        val adLoadListener = object : AdLoadListener<NativeAd> {
            override fun onAdLoaded(ads: List<NativeAd>) {
                Logger.info("successfully loaded nova native ads ...")
                val ad = ads[0].also { nativeAd = it }

                val mspAd = NovaNativeAd.Builder(this@NovaAdapter)
                    .title(ad.headline ?: "")
                    .body(ad.body ?: "")
                    .advertiser(ad.advertiser ?: "")
                    .callToAction(ad.callToAction ?: "")
                    .mediaView(MediaView(context))
                    .mediaController(NovaMediaController().also { ad.mediaListener = it })
                    .build()
                    .apply {
                        adInfo["price"] = winnerBid.price
                        adInfo["bidder_placement_id"] = adUnitId
                        adInfo["ad_loaded_at"] = System.currentTimeMillis()
                        adInfo["isVideo"] = ad.creativeType == NativeAd.CreativeType.VIDEO
                    }

                ad.setAdListener(object : NativeAdListener {
                    override fun onAdImpressed() {
                        mesTracker?.trackAdImpression(adRequest, winnerBid, mspAd)
                        handleAdImpression(adListener, adRequest)
                    }
                    override fun onAdClicked() {
                        handleAdClicked(adListener, adRequest)
                    }
                })

                handleAdLoaded(mspAd, adListener, adRequest)
            }

            override fun onAdLoadFailed(error: AdError) {
                Logger.info("failed to load nova native ads ...${error.message}")
                adListener.onError("Failed to load Nova native Ad: ${error.message}")
            }
        }

        val adLoader = AdLoader.Builder(context, adUnitId)
            .forNativeAd(adLoadListener)
            .build()

        adLoader?.loadAd(
            AdRequest.Builder(winnerBid.impId)
                .setAdString(winnerBid.adm)
                .build()
        )
    }

    override fun prepareViewForInteraction(
        nativeAd: MspNativeAd,
        nativeAdView: Any
    ) {
        if (nativeAdView !is MspNativeAdView) {
            return
        }

        val adView = NativeAdView(nativeAdView.context)
            .apply {
                advertiserView = nativeAdView.AdvertiserView
                headlineView = nativeAdView.adTitleView
                bodyView = nativeAdView.adBodyView
                mediaView = nativeAd.mediaView as? MediaView
                callToActionView = nativeAdView.AdCallToActionView
            }
        Utils.moveChildren(nativeAdView, adView)
        nativeAdView.addView(adView)
        this.nativeAd?.let { adView.setNativeAd(it) }
    }

    companion object {
        private var mesTracker: MesTracker? = null

        fun getOriginalNovaAdUnitId(winner: Bid): String? {
            return try {
                JSONObject(winner.jsonString)
                    .optJSONObject("ext")
                    ?.optJSONObject("nova")
                    ?.optString("ad_unit_id")
            } catch (ignored: JSONException) {
                null
            }
        }
    }

    class NovaInterstitialAd(novaAdapter: NovaAdapter, private val interstitialAd: InterstitialAd)
        : com.particles.msp.api.InterstitialAd(novaAdapter) {
        override fun show(activity: Activity) {
            interstitialAd.show(activity)
        }
    }
}