package com.particles.novaadapter

import android.app.Activity
import android.content.Context
import com.particles.android.ads.AdError
import com.particles.android.ads.AdFeatures
import com.particles.android.ads.AdLoadListener
import com.particles.android.ads.AdLoader
import com.particles.android.ads.AdRequest
import com.particles.android.ads.AdSize
import com.particles.android.ads.NovaSdk
import com.particles.android.ads.banner.BannerAd
import com.particles.android.ads.banner.BannerAdListener
import com.particles.android.ads.internal.NativeAdImpl
import com.particles.android.ads.internal.util.PlayableUtils
import com.particles.android.ads.interstitial.InterstitialAd
import com.particles.android.ads.interstitial.InterstitialAdListener
import com.particles.android.ads.nativead.MediaView
import com.particles.android.ads.nativead.NativeAd
import com.particles.android.ads.nativead.NativeAdListener
import com.particles.android.ads.nativead.NativeAdView
import com.particles.msp.adapter.AdNetwork
import com.particles.msp.adapter.AdNetworkAdapter
import com.particles.msp.adapter.AdapterInitListener
import com.particles.msp.adapter.AdapterInitStatus
import com.particles.msp.adapter.InitializationParameters
import com.particles.msp.api.AdFormat
import com.particles.msp.api.MSPConstants
import com.particles.msp.auction.AuctionBidListener
import com.particles.msp.util.Logger
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.json.JSONException
import org.json.JSONObject
import org.prebid.mobile.rendering.bidding.data.bid.Bid
import org.prebid.mobile.rendering.bidding.data.bid.BidResponse
import com.particles.msp.api.AdListener as MspAdListener
import com.particles.msp.api.AdRequest as MspAdRequest
import com.particles.msp.api.BannerAdView as MspBannerAdView
import com.particles.msp.api.NativeAd as MspNativeAd
import com.particles.msp.api.NativeAdView as MspNativeAdView


class NovaAdapter : AdNetworkAdapter() {
    private var nativeAd: NativeAd? = null
    private var bannerAd: BannerAd? = null
    private var interstitialAd: InterstitialAd? = null
    private val tagPrefix = "[Adapter: Nova]"

    override val metadata = Metadata(
        sdkVersion = NovaSdk.version
    )

    override val adNetworkName: String = "nova"
    override val adNetwork: AdNetwork = AdNetwork.Nova

    override fun destroyAd() {
        nativeAd?.destroy()
        nativeAd = null
        bannerAd = null
        interstitialAd = null
    }

    override fun initialize(
        initParams: InitializationParameters,
        adapterInitListener: AdapterInitListener,
        context: Context
    ) {
        super.initialize(initParams, adapterInitListener, context)
        CoroutineScope(Dispatchers.IO).launch {
            NovaSdk.initialize(context, initParams.getNovaServerApiHost()) {
                adapterInitListener.onComplete(
                    AdNetwork.Nova,
                    AdapterInitStatus.SUCCESS,
                    AdapterInitStatus.SUCCESS.message
                )
            }
        }
    }

    override fun loadAdCreative(
        bidResponse: BidResponse?,
        auctionBidListener: AuctionBidListener,
        context: Context,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
        bidderPlacementId: String,
        bidderInfo: com.particles.msp.auction.BidderInfo
    ) {
        super.loadAdCreative(bidResponse, auctionBidListener, context, adRequest, adListener, bidderPlacementId, bidderInfo)

        val winnerBid = bidResponse?.winningBid
        if (winnerBid == null) {
            handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load Nova Ad: No Winning bid returned")
            return
        }

        val adUnitId = getOriginalNovaAdUnitId(winnerBid)
        if (adUnitId == null) {
            handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load Nova Ad: failed to parse original ad unit id")
            return
        }

        when (adRequest.adFormat) {
            AdFormat.BANNER -> loadBannerAd(context, adUnitId, winnerBid, adRequest, adListener, auctionBidListener, bidderPlacementId)
            AdFormat.NATIVE -> loadNativeAd(context, adUnitId, winnerBid, adRequest, adListener, auctionBidListener, bidderPlacementId)
            AdFormat.INTERSTITIAL -> loadInterstitialAd(context, adUnitId, winnerBid, adRequest, adListener, auctionBidListener, bidderPlacementId)
            AdFormat.MULTI_FORMAT -> when (val type = winnerBid.prebid.type) {
                "banner" -> loadBannerAd(context, adUnitId, winnerBid, adRequest, adListener, auctionBidListener, bidderPlacementId)
                "native" -> loadNativeAd(context, adUnitId, winnerBid, adRequest, adListener, auctionBidListener, bidderPlacementId)
                else -> handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load Nova Ad: Unsupported Ad type: $type")
            }
            else -> handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load Nova Ad: Unsupported Ad Format: ${adRequest.adFormat}")
        }
    }

    private fun loadBannerAd(
        context: Context,
        adUnitId: String,
        winnerBid: Bid,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
        auctionBidListener: AuctionBidListener,
        bidderPlacementId: String
    ) {
        val adLoadListener = object : AdLoadListener<BannerAd> {
            override fun onAdLoaded(ads: List<BannerAd>) {
                Logger.info("$tagPrefix successfully loaded nova banner ads ...")
                val ad = ads[0].also { bannerAd = it }

                val mspAd = MspBannerAdView(ad.getAdView(), this@NovaAdapter)

                ad.setAdListener(object : BannerAdListener {
                    override fun onAdImpressed() {
                        handleAdImpression(adListener, adRequest)
                    }
                    override fun onAdClicked() {
                        handleAdClicked(adListener, adRequest)
                    }
                })

                handleAdLoaded(mspAd, auctionBidListener, bidderPlacementId, adUnitId, winnerBid.price)
            }

            override fun onAdLoadFailed(error: AdError) {
                Logger.info("$tagPrefix failed to load nova banner ads ...${error.message}")
                handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load Nova Ad: ${error.message}")
            }
        }

        val adLoader = AdLoader.Builder(context, adUnitId)
            .forBannerAd(adLoadListener)
            .build()

        adLoader?.loadAd(
            AdRequest.Builder(winnerBid.impId)
                .setAdSize(adRequest.adSize?.let { AdSize(it.width, it.height) })
                .setAdString(winnerBid.adm)
                .build()
        )
    }

    private fun loadInterstitialAd(
        context: Context,
        adUnitId: String,
        winnerBid: Bid,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
        auctionBidListener: AuctionBidListener,
        bidderPlacementId: String
    ) {
        val adLoadListener = object : AdLoadListener<InterstitialAd>  {
            private fun onAdLoaded(interstitialAd: InterstitialAd) {
                val popupCtaEnabled = adRequest.customParams[MSPConstants.CUSTOM_PARAM_KEY_POPUP_CTA_ENABLED] as? Boolean
                if (popupCtaEnabled != null) {
                    AdFeatures.setPopupCtaEnabled(interstitialAd, popupCtaEnabled)
                }

                val mspInterstitialAd = NovaInterstitialAd(this@NovaAdapter, interstitialAd)
                interstitialAd.setAdListener(object : InterstitialAdListener {
                    override fun onAdImpressed() {
                        handleAdImpression(adListener, adRequest)
                    }

                    override fun onAdClicked() {
                        handleAdClicked(adListener, adRequest)
                    }

                    override fun onAdDismissed() {
                        handleAdDismissed(adListener, adRequest)
                    }
                })
                handleAdLoaded(mspInterstitialAd, auctionBidListener, bidderPlacementId, adUnitId, winnerBid.price)
            }

            override fun onAdLoaded(ads: List<InterstitialAd>) {
                Logger.info("$tagPrefix successfully loaded nova interstitial ads ...")
                val ad = ads[0].also { interstitialAd = it }
                onAdLoaded(ad)
            }

            override fun onAdLoadFailed(error: AdError) {
                Logger.info("$tagPrefix Failed to load Nova interstitial Ad: ${error.message}")
                handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load Nova interstitial Ad: ${error.message}")
            }
        }

        val adLoader = AdLoader.Builder(context, adUnitId)
            .forInterstitialAd(adLoadListener)
            .build()

        adLoader?.loadAd(
            AdRequest.Builder(winnerBid.impId)
                .setAdSize(adRequest.adSize?.let { AdSize(it.width, it.height) })
                .setAdString(winnerBid.adm)
                .build()
        )
    }

    private fun loadNativeAd(
        context: Context,
        adUnitId: String,
        winnerBid: Bid,
        adRequest: MspAdRequest,
        adListener: MspAdListener,
        auctionBidListener: AuctionBidListener,
        bidderPlacementId: String
    ) {
        val adLoadListener = object : AdLoadListener<NativeAd> {
            override fun onAdLoaded(ads: List<NativeAd>) {
                Logger.info("$tagPrefix successfully loaded nova native ads ...")
                val ad = ads[0].also { nativeAd = it }

                val playInlineEnabled = adRequest.customParams[MSPConstants.CUSTOM_PARAM_KEY_VIDEO_PLAY_INLINE_ENABLED] as? Boolean
                if (playInlineEnabled != null) {
                    AdFeatures.setVideoInlinePlaybackEnabled(ad, playInlineEnabled)
                }
                val popupCtaEnabled = adRequest.customParams[MSPConstants.CUSTOM_PARAM_KEY_POPUP_CTA_ENABLED] as? Boolean
                if (popupCtaEnabled != null) {
                    AdFeatures.setPopupCtaEnabled(ad, popupCtaEnabled)
                }

                val mspAd = NovaNativeAd.Builder(this@NovaAdapter)
                    .title(ad.headline ?: "")
                    .body(ad.body ?: "")
                    .advertiser(ad.advertiser ?: "")
                    .iconUrl(ad.icon?.uri)
                    .callToAction(ad.callToAction ?: "")
                    .mediaView(MediaView(context).apply {
                        val useController = adRequest.customParams[MSPConstants.CUSTOM_PARAM_KEY_USE_MEDIA_CONTROLLER] as? Boolean
                        if (useController != null) {
                            setUseController(useController)
                        }
                    })
                    .mediaController(NovaMediaController().also { ad.mediaListener = it })
                    .build()
                    .apply {
                        adInfo["isVideo"] = ad.creativeType == NativeAd.CreativeType.VIDEO
                        (ad as? NativeAdImpl)?.let {
                            adInfo["nova_creative_type"] = it.creative.type
                            adInfo["nova_layout"] = it.creative.mediaLayoutType

                            val config = ad.abConfig.get("immersive_playable_ui")
                            if (!config.isNullOrEmpty() && PlayableUtils.isPlayableAd(ad)) {
                                adInfo["nova_immersive_playable_ui"] = config
                                adInfo["nova_playable_url"] = it.creative.playable?.url ?: ""
                            }
                        }
                    }

                ad.setAdListener(object : NativeAdListener {
                    override fun onAdImpressed() {
                        handleAdImpression(adListener, adRequest)
                    }
                    override fun onAdClicked() {
                        handleAdClicked(adListener, adRequest)
                    }
                })

                handleAdLoaded(mspAd, auctionBidListener, bidderPlacementId, adUnitId, winnerBid.price)
            }

            override fun onAdLoadFailed(error: AdError) {
                Logger.info("$tagPrefix failed to load nova native ads ...${error.message}")
                handleAdLoadError(auctionBidListener, bidderPlacementId, "Failed to load Nova native Ad: ${error.message}")
            }
        }

        val adLoader = AdLoader.Builder(context, adUnitId)
            .forNativeAd(adLoadListener)
            .build()

        adLoader?.loadAd(
            AdRequest.Builder(winnerBid.impId)
                .setAdString(winnerBid.adm)
                .build()
        )
    }

    override fun prepareViewForInteraction(
        nativeAd: MspNativeAd,
        nativeAdView: Any
    ) {
        if (nativeAdView !is MspNativeAdView) {
            return
        }

        val adView = NativeAdView(nativeAdView.context)
            .apply {
                advertiserView = nativeAdView.AdvertiserView
                headlineView = nativeAdView.adTitleView
                bodyView = nativeAdView.adBodyView
                mediaView = nativeAd.mediaView as? MediaView
                callToActionView = nativeAdView.AdCallToActionView
                iconView = nativeAdView.AdIconView
            }
        Utils.moveChildren(nativeAdView, adView)
        nativeAdView.addView(adView)
        this.nativeAd?.let { adView.setNativeAd(it) }
    }

    companion object {
        fun getOriginalNovaAdUnitId(winner: Bid): String? {
            return try {
                JSONObject(winner.jsonString)
                    .optJSONObject("ext")
                    ?.optJSONObject("nova")
                    ?.optString("ad_unit_id")
            } catch (ignored: JSONException) {
                null
            }
        }
    }

    class NovaInterstitialAd(novaAdapter: NovaAdapter, private val interstitialAd: InterstitialAd)
        : com.particles.msp.api.InterstitialAd(novaAdapter) {
        override fun show(activity: Activity) {
            interstitialAd.show(activity)
        }
    }
}