package com.particles.android.ads.browser

import android.graphics.Bitmap
import android.net.Uri
import android.os.Bundle
import android.view.MenuItem
import android.view.View
import android.view.ViewGroup
import android.view.ViewStub
import android.view.Window
import android.webkit.RenderProcessGoneDetail
import android.webkit.WebChromeClient
import android.webkit.WebResourceRequest
import android.webkit.WebView
import android.webkit.WebViewClient
import android.widget.ProgressBar
import android.widget.TextView
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.widget.Toolbar
import androidx.core.view.NestedScrollingChild
import com.google.android.material.appbar.AppBarLayout
import com.particles.android.ads.R
import com.particles.android.ads.internal.util.WebViews

class BrowserActivity : AppCompatActivity() {

    private lateinit var appBar: AppBarLayout
    private lateinit var toolbar: Toolbar
    private lateinit var toolbarBack: View
    private lateinit var toolbarClose: View
    private lateinit var toolbarTitle: TextView
    private lateinit var webView: WebView
    private lateinit var webProgressBar: ProgressBar
    private lateinit var loading: View
    private var floatingStub: ViewStub? = null
    private var floatingView: View? = null
    private var tracker: BrowserTracker? = null
    private var mediaHeader: BrowserMediaHeader? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        applyTheme()
        super.onCreate(savedInstanceState)
        supportRequestWindowFeature(Window.FEATURE_NO_TITLE)
        mediaHeader = BrowserMediaHeader.obtain(intent)
        if (mediaHeader?.floating == false) {
            if (/* TODO: AdsFeatureGating.isEnableNovaBrowserNewClose() */ false) {
                setContentView(R.layout._nova_activity_ads_browser_with_media_header_new_close)
            } else {
                setContentView(R.layout._nova_activity_ads_browser_with_media_header)
            }
        } else {
            setContentView(R.layout._nova_activity_ads_browser)
        }
        setupActionBar()
        setupWebView()
        setupLoading()
        if (mediaHeader?.floating == true) {
            setupFloating()
            (floatingView as? ViewGroup)?.let { mediaHeader?.attach(it) }
        }

        val url = intent.getStringExtra("url")
        val requestId = intent.getStringExtra("request_id")
        val adId = intent.getStringExtra("ad_id")
        val adUnitId = intent.getStringExtra("ad_unit_id")
        val adClickedAt = intent.getLongExtra("ad_clicked_at", System.currentTimeMillis())
        val adToken = intent.getStringExtra("ad_token")

        if (requestId != null && adId != null && adUnitId != null && adToken != null) {
            tracker = BrowserTracker(requestId, adId, adUnitId, adClickedAt, adToken)
        }

        title = url

        if (WebViews.isDeeplink(url)) {
            DeeplinkHandler {
                showLoading()
                tracker?.logPageStart()
                webView.loadUrl(it)
            }.handle(this, Uri.parse(url))
            if (webView.url.isNullOrEmpty()) {
                finish()
            }
        } else if (url != null) {
            showLoading()
            tracker?.logPageStart()
            webView.loadUrl(url)
        }
    }

    private fun applyTheme() {
//      TODO:
//      ThemeHelper.applyTheme(this)
        window.statusBarColor = getColor(R.color._nova_browser_bg)
    }

    private fun setupActionBar() {
        val appBarViewGroup = findViewById<AppBarLayout>(R.id.app_bar)
        if (appBarViewGroup != null) {
            appBarViewGroup.stateListAnimator = null
            mediaHeader?.attach(appBarViewGroup)
        }
        appBar = appBarViewGroup
        toolbar = findViewById(R.id.toolbar)
        toolbarBack = toolbar.findViewById(R.id.toolbar_back)
        toolbarClose = toolbar.findViewById(R.id.toolbar_close)
        toolbarTitle = toolbar.findViewById(R.id.toolbar_title)
        toolbarBack.setOnClickListener { onBackPressed() }
        toolbarClose.setOnClickListener {
            finish()
            tracker?.logPageClose("close", WebViews.getPageIndex(webView), WebViews.getScrollDepth(webView))
        }
        setSupportActionBar(toolbar)

        if (mediaHeader?.floating == false && /* TODO: AdsFeatureGating.isEnableNovaBrowserNewClose() */ false ) {
            val headerBack: View? = findViewById(R.id.header_back)
            val headerClose: View? = findViewById(R.id.header_close)
            headerBack?.setOnClickListener { toolbarBack.performClick() }
            headerClose?.setOnClickListener { toolbarClose.performClick() }
            appBar.addOnOffsetChangedListener { view, offset ->
                val translationY = toolbar.height - view.height - offset
                toolbarBack.translationY = translationY.toFloat()
                toolbarClose.translationY = translationY.toFloat()
            }
        }
    }

    private fun setupWebView() {
        webProgressBar = findViewById(R.id.progress_bar)
        webView = findViewById(R.id.web_view)
        webView.webChromeClient = object : WebChromeClient() {
            override fun onProgressChanged(view: WebView?, newProgress: Int) {
                webProgressBar.progress = newProgress
                webProgressBar.visibility = if (newProgress < 100) View.VISIBLE else View.INVISIBLE
                if (newProgress >= 100) {
                    hideLoading()
                    tracker?.logPageLoaded()
                    if (mediaHeader != null) {
                        setNestedScrollingEnabled(view?.canGoBack() != true)
                    }
                }
            }

            override fun onReceivedTitle(view: WebView?, title: String?) {
                if (title != null) {
                    setTitle(title)
                }
            }
        }
        webView.webViewClient = object : WebViewClient() {
            val deeplinkHandler = DeeplinkHandler { webView.loadUrl(it) }

            override fun shouldOverrideUrlLoading(
                view: WebView?,
                request: WebResourceRequest?
            ): Boolean {
                return view != null && request != null
                        && deeplinkHandler.handle(this@BrowserActivity, request.url)
            }

            override fun onPageCommitVisible(view: WebView?, url: String?) {
                super.onPageCommitVisible(view, url)
                hideLoading()
                tracker?.logTextLoaded()
            }

            override fun onPageStarted(view: WebView?, url: String?, favicon: Bitmap?) {
                super.onPageStarted(view, url, favicon)
                if (mediaHeader != null) {
                    setNestedScrollingEnabled(view?.canGoBack() != true)
                }
            }

            override fun onPageFinished(view: WebView?, url: String?) {
                super.onPageFinished(view, url)
                title = view?.title ?: url
            }

            override fun onRenderProcessGone(view: WebView?, detail: RenderProcessGoneDetail?): Boolean {
                return WebViews.onRenderProcessGone(view, detail)
            }
        }
    }

    private fun setupFloating() {
        floatingStub = findViewById(R.id.floating_stub)
        floatingView = floatingStub?.inflate()
        floatingView?.findViewById<View>(R.id.exo_position)?.visibility = View.INVISIBLE
        floatingView?.findViewById<View>(R.id.exo_progress)?.visibility = View.INVISIBLE
        floatingView?.findViewById<View>(R.id.exo_duration)?.visibility = View.INVISIBLE
        floatingView?.findViewById<View>(R.id.floating_dismiss)?.setOnClickListener {
            floatingView?.let {
                (it.parent as? ViewGroup)?.removeView(it)
            }
        }
    }

    private fun setupLoading() {
        loading = findViewById(R.id.loading)
    }

    private fun showLoading() {
        loading.visibility = View.VISIBLE
    }

    private fun hideLoading() {
        loading.visibility = View.GONE
    }

    private fun setNestedScrollingEnabled(enabled: Boolean) {
        (webView as? NestedScrollingChild)?.isNestedScrollingEnabled = enabled
        if (!enabled) {
            appBar.setExpanded(false)
        }
    }

    override fun setTitle(title: CharSequence?) {
        toolbarTitle.text = title
    }

    override fun onPause() {
        super.onPause()
        tracker?.logPagePause(WebViews.getPageIndex(webView), WebViews.getScrollDepth(webView))
    }

    override fun onResume() {
        super.onResume()
        tracker?.logPageResume()
    }

    override fun onBackPressed() {
        if (webView.canGoBack()) {
            webView.goBack()
            toolbarClose.visibility = View.VISIBLE
            if (mediaHeader?.floating == false && /* TODO: AdsFeatureGating.isEnableNovaBrowserNewClose() */ false) {
                findViewById<View>(R.id.header_close)?.visibility = View.VISIBLE
            }
        } else {
            super.onBackPressed()
            tracker?.logPageClose("go_back", WebViews.getPageIndex(webView), WebViews.getScrollDepth(webView))
        }
    }

    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        when (item.itemId) {
            android.R.id.home -> {
                onBackPressed()
                return true
            }
        }
        return super.onOptionsItemSelected(item)
    }

    override fun onDestroy() {
        super.onDestroy()
        webView.loadUrl("about:blank")
        webView.destroy()
    }

}
