package com.particles.android.ads.internal.loader

import android.content.Context
import com.particles.android.ads.AdError
import com.particles.android.ads.AdRequest
import com.particles.android.ads.internal.AdErrorPool
import com.particles.android.ads.internal.NovaInternal
import com.particles.android.ads.internal.data.mapper.AdEntityDataMapper
import com.particles.android.ads.internal.data.mapper.AdEntityJsonMapper
import com.particles.android.ads.internal.domain.Ad
import com.particles.android.ads.internal.domain.DedupeRecorder
import com.particles.android.ads.internal.util.Callback
import com.particles.android.ads.internal.util.DeviceInfo
import com.particles.android.ads.internal.util.OkHttpHolder
import com.particles.android.ads.internal.util.TestModeUtils
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.flowOn
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.launch
import okhttp3.HttpUrl
import okhttp3.HttpUrl.Companion.toHttpUrl
import okhttp3.Request
import org.json.JSONException
import org.json.JSONObject

class GetAdsTask2(
    private val context: Context,
    private val format: String,
    private val adUnitId: String,
    private val adRequest: AdRequest,
    private val maxNumberOfAds: Int,
) {
    private val baseUrl = NovaInternal.apiServer + "/ads"

    fun getAds(callback: Callback<List<Ad>>) {
        CoroutineScope(Dispatchers.Main).launch {
            flow {
                val response = if (!adRequest.adString.isNullOrEmpty()) {
                    adRequest.adString
                } else if (adRequest.testMode) {
                    TestModeUtils.getTestResponse(context, format)
                } else {
                    fetch(buildHttpUrl())
                }
                emit(response)
            }
                .map { parseResponse(it!!) }
                .flowOn(Dispatchers.IO)
                .catch { callback.onFailure(it) }
                .collect { callback.onSuccess(it) }
        }
    }

    private fun fetch(url: HttpUrl): String? {
        val request = Request.Builder().url(url).header("User-Agent", OkHttpHolder.httpAgent).build()
        val response = OkHttpHolder.okHttpClient.newCall(request).execute()
        response.use {
            return response.body?.string()
        }
    }

    private fun buildHttpUrl(): HttpUrl {
        return baseUrl.toHttpUrl().newBuilder()
            .addQueryParameter(ApiParamKey.FORMAT, format)
            .addQueryParameter(ApiParamKey.AD_UNIT, adUnitId)
            .addQueryParameter(ApiParamKey.REQ_ID, adRequest.id)
            .addQueryParameter(ApiParamKey.USER_ID, adRequest.userId)
            .addQueryParameter(ApiParamKey.PROFILE_ID, adRequest.profileId)
            .addQueryParameter(ApiParamKey.SESSION_ID, adRequest.sessionId)
            .addQueryParameter(ApiParamKey.TS, System.currentTimeMillis().toString())
            .apply {
                adRequest.deviceId?.let {
                    addQueryParameter(ApiParamKey.DEVICE_ID, it)
                }
                adRequest.weather?.let {
                    addQueryParameter(ApiParamKey.WEATHER, it)
                }
                adRequest.address?.postalCode?.let {
                    addQueryParameter(ApiParamKey.POSTAL_CODE, it)
                }
                adRequest.address?.locality?.let {
                    addQueryParameter(ApiParamKey.CITY, it)
                }
                adRequest.address?.adminArea?.let {
                    addQueryParameter(ApiParamKey.STATE, it)
                }
                adRequest.address?.locale?.let {
                    addQueryParameter(ApiParamKey.LANGUAGE, it.language)
                }
                adRequest.location?.let {
                    addQueryParameter(ApiParamKey.LATITUDE, it.latitude.toString())
                    addQueryParameter(ApiParamKey.LONGITUDE, it.longitude.toString())
                }
                adRequest.adSize?.let {
                    addQueryParameter(ApiParamKey.WIDTH, it.width.toString())
                    addQueryParameter(ApiParamKey.HEIGHT, it.height.toString())
                }
                adRequest.usPrivacy?.let {
                    addQueryParameter(ApiParamKey.US_PRIVACY, it)
                }
                adRequest.extras.forEach { (k, v) ->
                    addQueryParameter("x_$k", v.toString())
                }
            }
            .addQueryParameter(
                ApiParamKey.DEDUPE_INFO,
                DedupeRecorder.getRecordsToString(adUnitId, 3)
            )
            .addQueryParameter(ApiParamKey.NUM_ADS, maxNumberOfAds.toString())
            .apply {
                val di = DeviceInfo(context)
                addQueryParameter(ApiParamKey.MAKE, di.manufacturer)
                addQueryParameter(ApiParamKey.BRAND, di.brand)
                addQueryParameter(ApiParamKey.MODEL, di.model)
                addQueryParameter(ApiParamKey.OS, di.osName)
                addQueryParameter(ApiParamKey.OSV, di.osVersion)
                addQueryParameter(ApiParamKey.CARRIER, di.carrier)
                addQueryParameter(ApiParamKey.LANG, di.language)
                addQueryParameter(ApiParamKey.CT, di.connectionType)
                addQueryParameter(ApiParamKey.AAID, di.advertisingId)
                addQueryParameter(ApiParamKey.LMT, if (di.limitAdTrackingEnabled) "1" else "0")
                addQueryParameter(ApiParamKey.BUNDLE, context.packageName)
                addQueryParameter(ApiParamKey.CV, di.appVersion)
            }
            .build()
    }

    private fun parseResponse(response: String): List<Ad> {
        try {
            val json = JSONObject(response)

            val code = json.optInt("code", -1)
            if (code != 0) {
                val reason: String = json.optString("reason")
                throw AdError(code, reason)
            }

            val ad = json.getJSONArray("ad")
            val adEntities = AdEntityJsonMapper.map(ad)
            val ads = AdEntityDataMapper.map(adEntities)
            if (ads.isNotEmpty()) {
                return ads
            }

            throw AdErrorPool.NO_AD_FILLED.toAdError()
        } catch (e: JSONException) {
            throw AdErrorPool.GENERAL_FILL_ERROR.toAdError(e)
        }
    }
}
