package com.particles.android.ads.internal.util.viewability

import android.os.Handler
import android.os.Looper
import android.view.View
import android.view.ViewTreeObserver

private const val VIEWABILITY_THROTTLE_MILLIS = 100L

class ViewabilityTracker(
    private val view: View,
    private val handler: Handler = Handler(Looper.getMainLooper()),
    private val onViewabilityChanged: (state: ViewabilityState) -> Unit
) {
    private var isTracking = false
    private var isScheduled = false
    private val viewabilityChecker = ViewabilityChecker(view)
    private val onPreDrawListener = ViewTreeObserver.OnPreDrawListener {
        if (isTracking) {
            scheduleViewabilityCheck()
        }
        true
    }

    fun startTracking() {
        if (!isTracking) {
            isTracking = true

            val observer = view.viewTreeObserver
            if (observer != null && observer.isAlive) {
                observer.addOnPreDrawListener(onPreDrawListener)
            }

            scheduleViewabilityCheck()
        }
    }

    fun stopTracking() {
        if (isTracking) {
            isTracking = false

            val observer = view.viewTreeObserver
            if (observer != null && observer.isAlive) {
                observer.removeOnPreDrawListener(onPreDrawListener)
            }

            handler.removeMessages(0)
            isScheduled = false

            viewabilityChecker.reset()
        }
    }

    private fun scheduleViewabilityCheck() {
        if (isScheduled) {
            return
        }

        isScheduled = true
        handler.removeCallbacks(heartbeatRunnable)
        handler.postDelayed(viewabilityRunnable, VIEWABILITY_THROTTLE_MILLIS)
    }

    private val viewabilityRunnable = Runnable {
        isScheduled = false
        if (viewabilityChecker.hasChanged()) {
            onViewabilityChanged(viewabilityChecker.getState())
        }
        handler.postDelayed(heartbeatRunnable, 900)
    }

    private val heartbeatRunnable = Runnable {
        if (isTracking) {
            scheduleViewabilityCheck()
        }
    }
}
