package com.particles.android.ads

import android.os.Bundle
import android.view.View
import android.view.Window
import android.widget.ImageView
import android.widget.TextView
import androidx.appcompat.app.AppCompatActivity
import com.bumptech.glide.Glide
import com.particles.android.ads.internal.AppOpenAdImpl
import com.particles.android.ads.internal.BaseAdImpl
import com.particles.android.ads.internal.InterstitialAdImpl
import com.particles.android.ads.internal.NativeAdImpl
import com.particles.android.ads.internal.util.AdBank
import com.particles.android.ads.nativead.MediaView
import com.particles.android.ads.nativead.NativeAdListener
import com.particles.android.ads.nativead.NativeAdView

class AdActivity : AppCompatActivity() {

    companion object {
        const val EXTRA_AD_KEY = "ad_key"
    }

    private var adView: NativeAdView? = null
    private var adIconView: ImageView? = null
    private var advertiserView: TextView? = null
    private var adMediaView: MediaView? = null
    private var adHeadlineView: TextView? = null
    private var adBodyView: TextView? = null
    private var adCallToActionView: TextView? = null
    private var adDismissView: View? = null
    private var adVolumeOnView: View? = null
    private var adVolumeOffView: View? = null

    private var ad: BaseAdImpl? = null

    private var shouldAutoClose = false

    override fun onCreate(savedInstanceState: Bundle?) {
        supportRequestWindowFeature(Window.FEATURE_NO_TITLE)
        super.onCreate(savedInstanceState)

        val ad = intent.getStringExtra(EXTRA_AD_KEY)?.let { AdBank.get(it) }
        if (ad !is AppOpenAdImpl && ad !is InterstitialAdImpl) {
            finish()
            return
        }

        this.ad = ad as BaseAdImpl?

        val nativeAd = NativeAdImpl(ad.adSession).also {
            it.setAdListener(object: NativeAdListener {
                override fun onAdImpressed() {
                    ad.adListener?.onAdImpressed()
                }

                override fun onAdClicked() {
                    ad.adListener?.onAdClicked()
                }
            })
            it.setExtras(ad.extras)
        }

        val useVerticalLayout = nativeAd.isVerticalMedia

        if (useVerticalLayout) {
            setContentView(R.layout._nova_appopen_native_immersive_new)
        } else {
            setContentView(R.layout._nova_appopen_native_3_new)
        }

        adView = findViewById(R.id.ad_root)
        adIconView = findViewById(R.id.ad_icon)
        advertiserView = findViewById(R.id.advertiser)
        adMediaView = findViewById(R.id.ad_media)
        adHeadlineView = findViewById(R.id.ad_headline)
        adBodyView = findViewById(R.id.ad_body)
        adCallToActionView = findViewById(R.id.ad_call_to_action)
        adDismissView = findViewById(R.id.ad_dismiss)
        adVolumeOnView = findViewById(R.id.ad_volume_on)
        adVolumeOffView = findViewById(R.id.ad_volume_off)

        adVolumeOnView?.setOnClickListener {
            adMediaView?.findViewById<View>(R.id.player_controller_volume_on)?.let {
                it.performClick()
                setVolumeOnOff(false)
            }
        }
        adVolumeOffView?.setOnClickListener {
            adMediaView?.findViewById<View>(R.id.player_controller_volume_off)?.let {
                it.performClick()
                setVolumeOnOff(true)
            }
        }
        adMediaView?.setImageScaleType(ImageView.ScaleType.CENTER_CROP)
        if (useVerticalLayout) {
            adMediaView?.setUseController(false)
            this.ad?.adSession?.ad?.creative?.video?.isMutePlay?.let {
                setVolumeOnOff(!it)
            }
        }

        render(nativeAd)

        adDismissView?.setOnClickListener {
            ad.onAdDismissed("cancel")
            finish()
        }
    }

    private fun setVolumeOnOff(isOn: Boolean) {
        adVolumeOnView?.visibility = if (isOn) View.VISIBLE else View.GONE
        adVolumeOffView?.visibility = if (isOn) View.GONE else View.VISIBLE
    }

    private fun render(ad: NativeAdImpl) {
        adIconView?.let {
            val iconUri = ad.icon?.uri
            it.visibility = if (iconUri.isNullOrEmpty()) View.GONE else View.VISIBLE
            Glide.with(it.context.applicationContext).load(iconUri).circleCrop().into(it)
        }
        advertiserView?.text = ad.advertiser
        adHeadlineView?.text = ad.headline
        adBodyView?.text = ad.body
        adCallToActionView?.text = ad.callToAction

        adView?.advertiserView = advertiserView
        adView?.headlineView = adHeadlineView
        adView?.bodyView = adBodyView
        adView?.mediaView = adMediaView
        adView?.callToActionView = adCallToActionView
        adView?.setNativeAd(ad)
    }

    override fun onResume() {
        super.onResume()
        if (shouldAutoClose) {
            ad?.onAdDismissed("auto_close")
            finish()
        }
    }

    override fun onBackPressed() {
        super.onBackPressed()
        ad?.onAdDismissed("go_back")
    }

    override fun onDestroy() {
        super.onDestroy()
        adView?.setNativeAd(null)
    }
}