package com.particles.android.ads.browser

import android.content.ActivityNotFoundException
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.webkit.URLUtil
import java.net.URLEncoder

class DeeplinkHandler(private val redirectTo: (String) -> Unit) {
    fun handle(context: Context, uri: Uri): Boolean {
        if (URLUtil.isNetworkUrl(uri.toString())) {
            return false
        }

        try {
            val intent = Intent.parseUri(uri.toString(), Intent.URI_INTENT_SCHEME)

            val fallbackUrl = intent.getStringExtra("browser_fallback_url")
            val marketReferrer = intent.getStringExtra("market_referrer")
            intent.removeExtra("browser_fallback_url")
            intent.removeExtra("market_referrer")

            // 1. launch app by deeplink
            try {
                context.startActivity(intent)
                return true
            } catch (e: ActivityNotFoundException) {
                // ignore
            }

            // 2. open fallback url
            if (!fallbackUrl.isNullOrEmpty() && URLUtil.isHttpsUrl(fallbackUrl)) {
                redirectTo(fallbackUrl)
                return true
            }

            // 3. open play store
            val pkg = intent.`package`
            if (!pkg.isNullOrEmpty()) {
                val playStoreUrl = if (marketReferrer.isNullOrEmpty()) {
                    "https://play.google.com/store/apps/details?id=$pkg"
                } else {
                    "https://play.google.com/store/apps/details?id=$pkg&referrer=${URLEncoder.encode(marketReferrer)}"
                }
                try {
                    context.startActivity(Intent(Intent.ACTION_VIEW).apply {
                        data = Uri.parse(playStoreUrl)
                        setPackage("com.android.vending")
                    })
                } catch (e: ActivityNotFoundException) {
                    redirectTo(playStoreUrl)
                }
                return true
            }

            // 4. no-op
            return true
        } catch (ignored: Exception) {
            return false
        }
    }
}
