package com.particles.android.ads.internal.data.cache

import android.content.Context
import com.particles.android.ads.internal.data.mapper.AdEntityDataMapper
import com.particles.android.ads.internal.data.mapper.AdEntityJsonMapper
import com.particles.android.ads.internal.domain.Ad
import org.json.JSONObject
import java.io.File
import java.lang.Exception

class AdCacheImpl(private val rootDir: File) : AdCache {

    constructor(context: Context, name: String) : this(File(context.cacheDir, name))

    private fun getCacheDir(adUnitId: String): File {
        return File(rootDir, adUnitId)
    }

    internal fun getCacheAdFiles(dir: File): List<File> {
        return (dir.list() ?: emptyArray())
            .mapNotNull { it.toIntOrNull() }
            .sorted()
            .map { File(dir, it.toString()) }
    }

    override fun getAds(adUnitId: String, maxNumberOfAds: Int): List<Ad> {
        val dir = getCacheDir(adUnitId)

        val ads = mutableListOf<Ad>()
        getCacheAdFiles(dir).forEach {file ->
            if (ads.size >= maxNumberOfAds) return@forEach
            try {
                val text = file.readText()
                val adEntity = AdEntityJsonMapper.map(JSONObject(text))
                val ad = AdEntityDataMapper.map(adEntity)
                    .also { it.localCachePath = file.path }

                val now = System.currentTimeMillis()
                when {
                    ad.expireTimeMillis < now -> file.delete()
                    ad.startTimeMillis < now -> {
                        ads.add(ad)
                        file.delete()
                    }
                }
            } catch (e: Exception) {
                file.delete()
            }
        }
        return ads
    }

    override fun putAds(adUnitId: String, ads: List<Ad>) {
        val dir = getCacheDir(adUnitId)
        deleteFileRecursive(dir)
        dir.mkdirs()
        ads.forEachIndexed { index, ad ->
            try {
                File(dir, index.toString())
                    .also { ad.localCachePath = it.path }
                    .writeText(ad.raw)
            } catch (e: Exception) {
                // ignored
            }
        }
    }

    override fun removeAds(adUnitId: String) {
        val dir = getCacheDir(adUnitId)
        deleteFileRecursive(dir)
    }

    private fun deleteFileRecursive(file: File) {
        if (file.isDirectory) {
            file.listFiles()?.forEach {
                deleteFileRecursive(it)
            }
        }
        file.delete()
    }
}
