package com.particles.android.ads.internal.data.mapper

import com.particles.android.ads.internal.data.entity.AdEntity
import com.particles.android.ads.internal.data.entity.AddonItemEntity
import com.particles.android.ads.internal.data.entity.CarouselItemEntity
import com.particles.android.ads.internal.data.entity.CreativeEntity
import com.particles.android.ads.internal.data.entity.VideoItemEntity
import com.particles.android.ads.internal.domain.*

object AdEntityDataMapper : Mapper<AdEntity, Ad> {
    override fun map(source: AdEntity): Ad {
        return Ad(
            raw = source.raw,
            requestId = source.requestId,
            adId = source.adId,
            adSetId = source.adSetId,
            creative = CreativeEntityDataMapper.map(source.creative),
            price = source.price,
            startTimeMillis = source.startTimeMs,
            expireTimeMillis = source.expirationMs,
            encryptedAdToken = source.encryptedAdToken,
        )
    }
}

object CreativeEntityDataMapper : Mapper<CreativeEntity, Creative> {
    override fun map(source: CreativeEntity): Creative {
        return Creative(
            type = source.creativeType,
            adm = source.adm,
            headline = source.headline,
            body = source.body,
            icon = toImage(source.iconUrl),
            video = source.videoItem?.let { VideoEntityDataMapper.map(it) },
            addon = source.addonItem?.let { AddonEntityDataMapper.map(it) },
            images = listOfNotNull(toImage(source.imageUrl)),
            isImageClickable = source.isImageClickable,
            isVerticalMedia = when (source.creativeType) {
                "IMAGE" -> source.isVerticalImage
                "VIDEO" -> source.videoItem?.isVertical ?: false
                else -> false
            },
            starRating = 0.0,
            advertiser = source.advertiser,
            callToAction = source.callToAction,
            clickThroughUrl = source.ctrUrl,
            browserOption = toBrowserOption(source.launchOption),
            carouselItems = source.carouselItems.map { CarouselEntityDataMapper.map(it) },
            impressionTrackingUrls = source.thirdPartyImpressionTrackingUrls,
            clickTrackingUrls = source.thirdPartyClickTrackingUrls,
            adVerifications = source.thirdPartyViewTrackingUrls.map {
                AdVerification(javascriptResourceUrl = it)
            },
        )
    }

    private fun toBrowserOption(option: String): BrowserOption {
        return when (option) {
            "LAUNCH_WEBVIEW" -> BrowserOption.IN_APP
            "LAUNCH_BROWSER" -> BrowserOption.BROWSER
            else -> BrowserOption.BROWSER
        }
    }
}

object CarouselEntityDataMapper : Mapper<CarouselItemEntity, CarouselItem> {
    override fun map(source: CarouselItemEntity): CarouselItem {
        return CarouselItem(
            body = source.body,
            image = toImage(source.imageUrl),
            callToAction = source.callToAction,
            clickThroughUrl = source.ctrUrl,
        )
    }
}

object VideoEntityDataMapper : Mapper<VideoItemEntity, Video> {
    override fun map(source: VideoItemEntity): Video {
        return Video(
            videoUrl = source.videoUrl,
            coverUrl = source.coverUrl,
            isAutoPlay = source.isPlayAutomatically,
            isLoopPlay = source.isLoop,
            isMutePlay = source.isMute,
            isClickable = source.isVideoClickable,
            isContinuePlay = source.isPlayOnLandingPage,
        )
    }
}

object AddonEntityDataMapper : Mapper<AddonItemEntity, Addon?> {
    override fun map(source: AddonItemEntity): Addon? {
        return when (source.type) {
            "display_card" -> DisplayCardAddon(
                imageUrl = source.imageUrl,
                displayTime = source.displayTime,
            )

            else -> null
        }
    }
}

private fun toImage(url: String?): Image? {
    return if (url.isNullOrEmpty()) null else Image(url, 0, 0)
}
