package com.particles.android.ads.internal.domain

import com.particles.android.ads.AdRequest
import com.particles.android.ads.AdSize
import com.particles.android.ads.internal.tracking.EventTracker

class AdSession(
    val id: String,
    val ad: Ad,
    val adUnitId: String,
    val adSize: AdSize?,
    val userId: String?,
    val profileId: String?,
    val sessionId: String?,
    val testMode: Boolean,
) {
    constructor(adUnitId: String, adRequest: AdRequest, ad: Ad) : this(
        id = ad.requestId.ifEmpty { adRequest.id },
        ad = ad,
        adUnitId = adUnitId,
        adSize = adRequest.adSize,
        userId = adRequest.userId,
        profileId = adRequest.profileId,
        sessionId = adRequest.sessionId,
        testMode = adRequest.testMode,
    )

    val adPlaybackState = AdPlaybackState()

    var adImpressedAt: Long = 0L
        private set

    var adViewableImpressedAt: Long = 0L
        private set

    var adClickedAt: Long = 0L
        private set

    var isAdHidden: Boolean = false
        private set

    private var adEventTimes = mutableMapOf<String, Long>()

    init {
        DedupeRecorder.onAdFilled(this)
    }

    fun adEventTime(type: String): Long {
        return adEventTimes[type] ?: 0L
    }

    fun onAdImpressed(extras: Map<String, Any?>?) {
        adImpressedAt = System.currentTimeMillis()

        onAdEvent(
            TrackingEvent.EVENT_TYPE_IMPRESSION,
            time = adImpressedAt,
            extras = extras
        )

        DedupeRecorder.onAdImpression(this)
    }

    fun onAdViewableImpressed(adViewability: AdViewability, extras: Map<String, Any?>?) {
        adViewableImpressedAt = System.currentTimeMillis()
        onAdEvent(
            TrackingEvent.EVENT_TYPE_VIEWABLE_IMPRESSION,
            time = adViewableImpressedAt,
            viewabilityExtras = ViewabilityExtras(
                viewportSize = adViewability.viewportSize,
                timeOnScreenInMillis = adViewability.visibleTimes,
                creativePosition = adViewability.viewPosition,
            ),
            extras = extras
        )
    }

    fun onAdClicked(clickAreaName: String?, extras: Map<String, Any?>?) {
        adClickedAt = System.currentTimeMillis()
        onAdEvent(
            TrackingEvent.EVENT_TYPE_CLICK,
            time = adClickedAt,
            clickExtras = ClickExtras(
                clickAreaName = clickAreaName ?: "",
            ),
            extras = extras
        )
    }

    fun onAdDismissed(reason: String?, extras: Map<String, Any?>?) {
        onAdEvent(TrackingEvent.EVENT_TYPE_DISMISS_AD, reason = reason, extras = extras)
    }

    fun onAdHidden(reason: String?, extras: Map<String, Any?>?) {
        isAdHidden = true
        onAdEvent(TrackingEvent.EVENT_TYPE_HIDE_AD, reason = reason, extras = extras)
    }

    fun onAdUnhidden(extras: Map<String, Any?>?) {
        isAdHidden = false
        onAdEvent(TrackingEvent.EVENT_TYPE_UNHIDE_AD, extras = extras)
    }

    fun onAdEvent(
        type: String,
        time: Long = System.currentTimeMillis(),
        offset: Long = -1L,
        reason: String? = null,
        videoExtras: VideoExtras? = null,
        videoProgressExtras: VideoProgressExtras? = null,
        viewabilityExtras: ViewabilityExtras? = null,
        clickExtras: ClickExtras? = null,
        extras: Map<String, Any?>? = null,
    ) {
        val recordKey = when (type) {
            TrackingEvent.EVENT_TYPE_VIDEO_PROGRESS -> "$type-$offset"
            else -> type
        }
        adEventTimes[recordKey] = time

        if (testMode) {
            return
        }

        EventTracker.fire(TrackingEvent(
            time = time,
            type = type,
            offset = offset,
            reason = reason,
            userId = userId,
            profileId = profileId,
            sessionId = sessionId,
            adUnitId = adUnitId,
            encryptedAdToken = ad.encryptedAdToken,
            duration = time - adImpressedAt,
            videoExtras = videoExtras,
            videoProgressExtras = videoProgressExtras,
            viewabilityExtras = viewabilityExtras,
            clickExtras = clickExtras,
            extras = extras,
        ))
    }
}
