package com.particles.android.ads.internal.domain

import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import java.lang.ref.WeakReference
import java.util.*

object DedupeRecorder {

    private val records = LinkedList<WeakReference<AdSession>>()

    @Synchronized
    private fun findRecord(adSession: AdSession): WeakReference<AdSession>? {
        return records.find { it.get() == adSession }
    }

    @Synchronized
    private fun addRecord(record: WeakReference<AdSession>) {
        records.addFirst(record)
    }

    @Synchronized
    private fun removeRecord(record: WeakReference<AdSession>) {
        records.remove(record)
    }

    fun onAdFilled(adSession: AdSession) {
        val record = findRecord(adSession)
        if (record == null) {
            addRecord(WeakReference(adSession))
        }
    }

    fun onAdImpression(adSession: AdSession) {
        val record = findRecord(adSession)
        if (record != null) {
            CoroutineScope(Dispatchers.Main).launch {
                delay(3000)
                removeRecord(record)
            }
        }
    }

    @Synchronized
    fun getRecords(adUnitId: String, n: Int = 3): List<AdRecord> {
        records.removeAll { it.get() == null }
        return records.mapNotNull { it.get() }
            .map { AdRecord(it.adUnitId, it.ad.adId, it.ad.adSetId) }
            .sortedBy { if (it.adUnitId == adUnitId) 0 else 1 }
            .take(n)
    }

    fun getRecordsToString(adUnitId: String, n: Int = 3): String {
        return getRecords(adUnitId, n).joinToString(";") { "${it.adUnitId},${it.adId},${it.adSetId}" }
    }

}

data class AdRecord(
    val adUnitId: String,
    val adId: String,
    val adSetId: String,
)
